package ru.yandex.bannerstorage.messaging.utils;

import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.util.Objects;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.jetbrains.annotations.NotNull;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

/**
 * Вспомогательные классы для работы с xml DOM
 *
 * @author egorovmv
 */
public final class XmlHelper {
    private XmlHelper() {
    }

    public static DocumentBuilder createSimpleBuilder() {
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        factory.setCoalescing(true);
        factory.setIgnoringComments(true);
        factory.setNamespaceAware(true);
        factory.setValidating(false);
        try {
            return factory.newDocumentBuilder();
        } catch (ParserConfigurationException e) {
            throw new RuntimeException(e);
        }
    }

    public static Document parseSimpleDocument(@NotNull InputStream input) {
        DocumentBuilder builder = createSimpleBuilder();
        try {
            return builder.parse(input);
        } catch (IOException | SAXException e) {
            throw new RuntimeException(e);
        }
    }

    public static Document parseSimpleDocument(@NotNull String input) {
        DocumentBuilder builder = createSimpleBuilder();
        try {
            return builder.parse(new InputSource(new StringReader(input)));
        } catch (IOException | SAXException e) {
            throw new RuntimeException(e);
        }
    }

    @NotNull
    public static Stream<Node> toNodeStream(@NotNull NodeList nodeList) {
        Objects.requireNonNull(nodeList, "nodeList");
        return IntStream.range(0, nodeList.getLength())
                .mapToObj(nodeList::item);
    }

    @NotNull
    public static Stream<Element> toElementStream(@NotNull NodeList nodeList) {
        return toNodeStream(nodeList)
                .filter(n -> n.getNodeType() == Node.ELEMENT_NODE)
                .map(n -> (Element) n);
    }
}
