package ru.yandex;

import org.apache.commons.dbcp2.BasicDataSource;
import org.apache.tomcat.InstanceManager;
import org.apache.tomcat.SimpleInstanceManager;
import org.eclipse.jetty.annotations.ServletContainerInitializersStarter;
import org.eclipse.jetty.apache.jsp.JettyJasperInitializer;
import org.eclipse.jetty.jndi.NamingUtil;
import org.eclipse.jetty.plus.annotation.ContainerInitializer;
import org.eclipse.jetty.server.Server;
import org.eclipse.jetty.servlet.ServletHolder;
import org.eclipse.jetty.webapp.WebAppContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.servlet.DispatcherServlet;
import ru.yandex.utils.Environment;
import ru.yandex.utils.SslUtils;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import java.net.InetSocketAddress;
import java.util.ArrayList;
import java.util.List;

import static ru.yandex.utils.Environment.Type.DEVELOPMENT;

/**
 * @author elwood
 */
public class Application {
    private static final Logger logger = LoggerFactory.getLogger(Application.class);

    private static List<ContainerInitializer> jspInitializers() {
        JettyJasperInitializer sci = new JettyJasperInitializer();
        ContainerInitializer initializer = new ContainerInitializer(sci, null);
        List<ContainerInitializer> initializers = new ArrayList<>();
        initializers.add(initializer);
        return initializers;
    }

    public static void main(String[] args) {
        logger.info("Starting premoderation UI web app..");

        logger.info("Port: {}", PropertiesProvider.getHttpPort());
        logger.info("JdbcUrl: {}", PropertiesProvider.getJdbcUrl());
        logger.info("JdbcUsername: {}", PropertiesProvider.getJdbcUsername());
        logger.info("Path to webapp: {}", PropertiesProvider.getWebappPath());

        // Отключаем провеку сертификатов на разработке
        if (Environment.getType().equals(DEVELOPMENT)) {
            SslUtils.disableSslCheck();
        }

        // Listen on all interfaces to expose port with docker. Would not work with `localhost`
        Server server = new Server(new InetSocketAddress("0.0.0.0",
                Integer.parseInt(PropertiesProvider.getHttpPort())));

        WebAppContext context = new WebAppContext();
        context.setContextPath("/");
        context.setWar(PropertiesProvider.getWebappPath());
        context.setCopyWebDir(false);
        context.setCopyWebInf(false);
        context.setExtractWAR(false);

        context.setAttribute("org.eclipse.jetty.containerInitializers", jspInitializers());
        context.setAttribute(InstanceManager.class.getName(), new SimpleInstanceManager());

        ServletHolder servletHolder = new ServletHolder("dispatcher", DispatcherServlet.class);
        servletHolder.setInitOrder(1);
        context.addServlet(servletHolder, "/*");

        // This is required to enable TLDs processing for JSTL
        context.addBean(new ServletContainerInitializersStarter(context), true);

        BasicDataSource dataSource = new BasicDataSource();
        dataSource.setDriverClassName(PropertiesProvider.getJdbcDriverClassName());
        dataSource.setValidationQuery("select 1");
        dataSource.setUrl(PropertiesProvider.getJdbcUrl());
        dataSource.setUsername(PropertiesProvider.getJdbcUsername());
        dataSource.setPassword(PropertiesProvider.getJdbcPassword());
        dataSource.setMaxIdle(25);
        dataSource.setMaxTotal(50);
        // Set loginTimeout to get SocketTimeoutException on connect in predictable time (need for /ping)
        // socketTimeout регулирует максимальный глобальный таймаут для драйвера
        // выполнить запрос с таймаутом, превышающим это значение, не получится
        dataSource.setConnectionProperties("loginTimeout=1;socketTimeout=120");
        // таймаут запроса по умолчанию
        // для конкретного запроса его можно изменить в большую или меньшую сторону,
        // позвав statement.setQueryTimeout(), но значение не может превысить socketTimeout
        dataSource.setDefaultQueryTimeout(30);

        try {
            InitialContext ic = new InitialContext();
            Context env = ic.createSubcontext("java:comp/env");
            NamingUtil.bind(env, "jdbc/moderation", dataSource);
        } catch (NamingException e) {
            throw new RuntimeException(e);
        }

        server.setHandler(context);

        try {
            server.start();
            server.join();
        } catch (Exception e) {
            logger.error("Unexpected exception while running jetty", e);
        }
    }
}
