package ru.yandex;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.IDN;
import java.net.URI;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @author elwood
 */
public class Utils {
    private static final Logger logger = LoggerFactory.getLogger(Utils.class);

    private static final String METRICA_BINDING_TYPE = "METRICA_BINDING_TYPE";
    private static final String LINK_URL_ARGS = "LINK_URL_ARGS";

    private static final String SELECT_URLS_SQL =
            "select dbo.clr_uf_BeautifyUrl(url) url from vt_creative_version_url where len(url) > 0 and creative_version_nmb = ?" +
            " union all " +
            " select value url from vt_creative_version_parameter where parameter_name in ('MOB_LINK_') and creative_version_nmb = ?";

    private static final String SELECT_URLS_WITH_PIXELS_SQL =
            "select dbo.clr_uf_BeautifyUrl(url) url from vt_creative_version_url_with_pixel where len(url) > 0 and creative_version_nmb = ?" +
            " union all " +
            " select value url from vt_creative_version_parameter where parameter_name in ('MOB_LINK_') and creative_version_nmb = ?";

    private static final String IS_CREATIVE_FOR_AWAPS_SQL =
            "SELECT creative_for_awaps = CASE WHEN" +
            "  EXISTS(SELECT 1 FROM t_creative_version_dsp cvd WHERE cvd.creative_version_nmb = ? AND dsp_nmb = 1)" +
            "  AND NOT EXISTS (SELECT 1 FROM t_creative_version_dsp cvd WHERE cvd.creative_version_nmb = ? AND dsp_nmb <> 1)" +
            " THEN 1 ELSE 0 END";

    private static final String SELECT_YACLID_PARAMS_SQL = "SELECT p.name, int_value = pv.value_int, string_value = s.value" +
            " FROM t_creative_version cv" +
            "  JOIN t_creative_version_param_value pv ON pv.creative_version_nmb = cv.nmb" +
            "  JOIN t_parameter p ON pv.parameter_nmb = p.nmb" +
            "  LEFT JOIN t_string s ON pv.string_nmb = s.nmb" +
            " WHERE cv.nmb = ? AND p.name IN ('" + METRICA_BINDING_TYPE + "', '" + LINK_URL_ARGS + "')";

    private static final String YACLID_AWAPS = "yaclid=0000000000";

    private static final int METRICA_BINDING_TYPE_EXACT = 1;

    /**
     * Возвращает список ссылок для указанной версии креатива с учётом Yaclid-specific параметров.
     */
    public static List<String> getCreativeVersionURLs(Connection connection, int creativeVersionNmb, boolean includePixels) {
        Map<String, Object> yaclidParameters = getYaclidParameters(connection, creativeVersionNmb);

        String linkUrlArgs = null;
        if (creativeForAwaps(connection, creativeVersionNmb)) {
            // Добавляем yaclid-параметр только если выбран тип "Точная регистрация кликов"
            if (yaclidParameters.containsKey(METRICA_BINDING_TYPE)) {
                if (METRICA_BINDING_TYPE_EXACT == (Integer) yaclidParameters.get(METRICA_BINDING_TYPE)) {
                    linkUrlArgs = YACLID_AWAPS;
                }
            }
        } else {
            linkUrlArgs = (String) yaclidParameters.get(LINK_URL_ARGS);
        }

        List<String> list = new ArrayList<>();
        String query = includePixels ? SELECT_URLS_WITH_PIXELS_SQL : SELECT_URLS_SQL;
        try (PreparedStatement st = connection.prepareStatement(query)) {
            st.setInt(1, creativeVersionNmb);
            st.setInt(2, creativeVersionNmb);
            try (ResultSet resultSet = st.executeQuery()) {
                while (resultSet.next()) {
                    String url = resultSet.getString("url");
                    list.add(linkUrlArgs != null ? addLinkUrlArgs(url, linkUrlArgs) : url);
                }
            }
        } catch (SQLException e) {
            throw new RuntimeException(e);
        }
        return list;
    }

    private static final Pattern SCHEME_PATTERN = Pattern.compile("[\\w\\d\\+\\-\\.]+://.*");
    private static final Pattern AUTHORITY_PATTERN = Pattern.compile("(?<userinfo>[^@]*@)?(?<host>[^:]*)(?<port>:[\\d]*)?");

    public static String addLinkUrlArgs(String url, String linkUrlArgs) {
        try {
            // Урлы, начинающиеся с двойного слеша (protocol-relative urls), должны тоже обрабатываться корректно
            // Для этого мы для них добавляем временно схему http, а после проверки и нормализации уберём снова
            // (эти манипуляции нужны, потому что URI сам по себе не переваривает такие урлы)
            boolean isProtocolRelative = url.startsWith("//");

            // Определяем, есть ли у урла схема
            boolean hasScheme = SCHEME_PATTERN.matcher(url).matches();

            // В .NET-овском UriBuilder в конструкторе делается такая же штука
            URI uri = new URI(isProtocolRelative ? "http:" + url
                    : (hasScheme ? url : "http://" + url));

            String authority = uri.getAuthority();

            String host;
            int port;
            if (uri.getHost() != null) {
                // Если URI смог самостоятельно распарсить host - используем его.
                host = uri.getHost();
                port = uri.getPort();
            } else {
                // Иначе парсим самостоятельно из authority. Эти приседания нужны из-за того,
                // что URI не умеет работать с нелатинскими доменами. Но URI корректно выделяет authority
                // Пример такого урла: http://сайт.рф:80/path/f?q=1&q=2&#!hashBang=452
                Matcher matcher = AUTHORITY_PATTERN.matcher(authority);
                if (!matcher.matches()) return null;
                host = matcher.group("host");
                String strPort = matcher.group("port");
                port = strPort != null && !strPort.isEmpty() ? Integer.parseInt(strPort.substring(1)) : -1;
            }

            // Выполняем преобразование "сайт.рф" в "xn--80aswg.xn--p1ai"
            // Первый вызов этого метода очень медленный из-за особенностей реализации в JDK (около 300мс)
            // Последующие вызовы выполняются быстро.
            String idnHost = IDN.toASCII(host);

            // Здесь собираем новый URI на базе старого.
            // Если в исходном URI был явно указан 80 порт, мы его уберём. userinfo из authority тоже убираем.
            // Схему переводим в нижний регистр
            URI idnUri = new URI(buildUriString(
                    uri.getScheme().toLowerCase(),
                    idnHost,
                    port != 80 ? port : -1,
                    uri.getRawPath(),
                    null != uri.getRawQuery() && !uri.getRawQuery().isEmpty() ? linkUrlArgs + "&" + uri.getRawQuery() : linkUrlArgs,
                    uri.getRawFragment()));

            // Метод toASCIIString() выполнит percent-encoding для всех unsafe символов
            String result = idnUri.toASCIIString();

            if (isProtocolRelative) {
                if (!result.startsWith("http://")) throw new AssertionError("This shouldn't happen");
                return result.substring("http:".length());
            }
            return result;
        } catch (Exception e) {
            // Не удалось добавить - пишем в лог ошибку и возвращаем оригинальный урл
            logger.error(String.format("Error when adding link URL args '%s' to url '%s'", linkUrlArgs, url));
            return url;
        }
    }

    /**
     * Собирает строку для создания URI из отдельно указанных компонент.
     * @param scheme Схема, обязательна
     * @param host Имя хоста, обязательно
     * @param port Порт, -1 если в результирующей строке его не нужно указывать
     * @param path Путь, необязателен
     * @param query Запрос, необязателен
     * @param fragment Фрагмент, необязателен
     */
    private static String buildUriString(String scheme, String host, int port, String path, String query, String fragment) {
        StringBuilder sb = new StringBuilder();
        sb.append(scheme).append("://");
        sb.append(host);
        if (port != -1)
            sb.append(":").append(Integer.toString(port));
        if (path != null)
            sb.append(path);
        if (query != null || fragment != null)
            sb.append("?");
        if (query != null)
            sb.append(query);
        if (fragment != null)
            sb.append("#").append(fragment);
        return sb.toString();
    }

    /**
     * Возвращает true, если на креативе будет срабатывать заточка под AWAPS
     * (если только одно DSP на креативе и это DSP - AWAPS) и false в противном случае.
     */
    private static boolean creativeForAwaps(Connection connection, int creativeVersionNmb) {
        try (PreparedStatement st = connection.prepareStatement(IS_CREATIVE_FOR_AWAPS_SQL)) {
            st.setInt(1, creativeVersionNmb);
            st.setInt(2, creativeVersionNmb);
            try (ResultSet resultSet = st.executeQuery()) {
                resultSet.next();
                return resultSet.getBoolean("creative_for_awaps");
            }
        } catch (SQLException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * Возвращает Yaclid-specific параметры со значениями для указанной версии креатива.
     * В качестве значения METRICA_BINDING_TYPE может быть целое число, а LINK_URL_ARGS - строка.
     */
    private static Map<String, Object> getYaclidParameters(Connection connection, int creativeVersionNmb) {
        Map<String, Object> map = new HashMap<>();
        try (PreparedStatement st = connection.prepareStatement(SELECT_YACLID_PARAMS_SQL)) {
            st.setInt(1, creativeVersionNmb);
            try (ResultSet resultSet = st.executeQuery()) {
                while (resultSet.next()) {
                    String name = resultSet.getString("name");
                    Integer intValue = resultSet.getInt("int_value");
                    if (resultSet.wasNull()) intValue = null;
                    String stringValue = resultSet.getString("string_value");
                    if (resultSet.wasNull()) stringValue = null;
                    map.put(name, intValue != null ? intValue : stringValue);
                }
                return map;
            }
        } catch (SQLException e) {
            throw new RuntimeException(e);
        }
    }
}
