package ru.yandex.controllers;

import com.google.common.io.ByteStreams;
import net.yandex.bannerstorage.service.RedirectChain;
import net.yandex.bannerstorage.service.RotorClientFactory;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.util.EntityUtils;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.ModelMap;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import ru.yandex.CheckResult;
import ru.yandex.CheckerAsyncResult;
import ru.yandex.ModerationResult;
import ru.yandex.Utils;
import ru.yandex.exceptions.BadRequestException;
import ru.yandex.exceptions.NotFoundException;
import ru.yandex.models.DocumentsDTO;
import ru.yandex.models.ParameterValueDTO;
import ru.yandex.models.RejectReasonDTO;
import ru.yandex.models.SiteDTO;
import ru.yandex.services.CreativeService;
import ru.yandex.utils.SslUtils;

import javax.servlet.http.HttpServletRequest;
import javax.sql.DataSource;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.io.UnsupportedEncodingException;
import java.net.IDN;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.sql.*;
import java.util.*;
import java.util.concurrent.CompletableFuture;

import static java.nio.charset.StandardCharsets.UTF_8;
import static java.util.stream.Collectors.joining;
import static ru.yandex.utils.Environment.Type.DEVELOPMENT;

/**
 * @author elwood
 */
@Controller
@RequestMapping("creative")
public class CreativesController {
    private final Logger logger = LoggerFactory.getLogger(CreativesController.class);
    private final DataSource dataSource;
    private RotorClientFactory rotorClientFactory;
    private final CloseableHttpClient httpClient;
    private final CreativeService creativeService;

    @Autowired
    public CreativesController(DataSource dataSource, RotorClientFactory rotorClientFactory, CreativeService creativeService) {
        this.dataSource = dataSource;
        this.rotorClientFactory = rotorClientFactory;
        this.creativeService = creativeService;
        this.httpClient = buildHttpClient();
    }

    // todo : move to @Config-classes
    private CloseableHttpClient buildHttpClient() {
        if (ru.yandex.utils.Environment.getType().equals(DEVELOPMENT)) {
            return HttpClientBuilder.create().useSystemProperties()
                    .setSSLContext(SslUtils.buildInsecureSSLContext())
                    .setSSLHostnameVerifier(SslUtils.buildInsecureHostnameVerifier())
                    .setMaxConnPerRoute(100)
                    .setMaxConnTotal(100)
                    .build();
        } else {
            return HttpClientBuilder.create().useSystemProperties()
                    .setMaxConnPerRoute(100)
                    .setMaxConnTotal(100)
                    .build();
        }
    }

    private String getCodeUrl(int nmb) throws SQLException {
        try (Connection connection = dataSource.getConnection()) {
            try (PreparedStatement stmt = connection.prepareStatement(
                    "exec sp_creative_version_get_preview @creative_version_nmb = ?")) {
                stmt.setInt(1, nmb);
                try (ResultSet rs = stmt.executeQuery()) {
                    if (rs.next()) {
                        return rs.getString("url");
                    } else {
                        throw new NotFoundException();
                    }
                }
            }
        }
    }

    private List<String> getCreativeUrls(int creativeVersionNmb) throws SQLException {
        try (Connection connection = dataSource.getConnection()) {
            return Utils.getCreativeVersionURLs(connection, creativeVersionNmb, false);
        }
    }

    private List<String> getCreativeUrlsWithPixels(int creativeVersionNmb) throws SQLException {
        try (Connection connection = dataSource.getConnection()) {
            return Utils.getCreativeVersionURLs(connection, creativeVersionNmb, true);
        }
    }

    private String getUrlsDomain(URL url) {
        String host = IDN.toUnicode(url.getHost());
        if (host.startsWith("www.")) {
            return host.substring("www.".length());
        } else {
            return host;
        }
    }

    private String getCmd(int creativeVersionNmb) throws SQLException {
        String cmd_where, site_0 = "", site_1 = "", geo_0 = "", geo_1 = "", cmd_where_site = "", cmd_where_geo = "";
        try (Connection connection = dataSource.getConnection()) {
            try (PreparedStatement stmt = connection.prepareStatement(
                    "exec sp_moderation_creative_select 'site', @creative_version_nmb = ?")) {
                stmt.setInt(1, creativeVersionNmb);

                try (ResultSet rs = stmt.executeQuery()) {
                    while (rs.next()) {
                        int site_nmb = rs.getInt("site_nmb");
                        int exclude = rs.getInt("exclude");

                        if (exclude == 0)
                            site_1 = site_1 + (site_1.length() > 0 ? " or " : "") + " site like '%;" + site_nmb + ";%'";
                        else
                            site_0 = site_0 + (site_0.length() > 0 ? " and " : "") + " site not like '%;" + site_nmb + ";%'";
                    }
                }
            }
            if (site_1.length() > 0) {
                site_1 = "(" + site_1 + ") ";
                cmd_where_site = site_1;
            }
            if (site_0.length() > 0)
                cmd_where_site = cmd_where_site + (cmd_where_site.length() == 0 ? "" : " and ") + site_0;

            try (PreparedStatement stmt = connection.prepareStatement(
                    "exec sp_moderation_creative_select 'geo', @creative_version_nmb = ?")) {
                stmt.setInt(1, creativeVersionNmb);
                try (ResultSet rs = stmt.executeQuery()) {
                    while (rs.next()) {
                        int geo_nmb = rs.getInt("geo_nmb");
                        int exclude = rs.getInt("exclude");

                        if (exclude == 0)
                            geo_1 = geo_1 + (geo_1.length() > 0 ? " or " : "") + " geo like '%;" + geo_nmb + ";%'";
                        else
                            geo_0 = geo_0 + (geo_0.length() > 0 ? " and " : "") + " geo not like '%;" + geo_nmb + ";%'";
                    }
                }
            }
            if (geo_1.length() > 0) {
                geo_1 = "(" + geo_1 + ") ";
                cmd_where_geo = geo_1;
            }
            if (geo_0.length() > 0)
                cmd_where_geo = cmd_where_geo + (cmd_where_geo.length() == 0 ? "" : " and ") + geo_0;
            if (cmd_where_site.length() > 0) cmd_where_site = " ( " + cmd_where_site + " or len(site) = 0) ";
            if (cmd_where_geo.length() > 0) cmd_where_geo = " ( " + cmd_where_geo + " or len(geo) = 0) ";
            cmd_where = cmd_where_site;
            if (cmd_where_geo.length() > 0)
                cmd_where = cmd_where + (cmd_where.length() == 0 ? "" : " and ") + cmd_where_geo;
        }
        return cmd_where;
    }

    private CheckerAsyncResult beginCheckAbsoluteUrl(String absoluteUrl, RotorClientFactory rotorClientFactory) {
        List<String> failedUrlsList = new ArrayList<>();
        CompletableFuture<RedirectChain> future = null;
        try {
            future = rotorClientFactory.getClient().fetchUrlTrace(new URL(absoluteUrl));
        } catch (MalformedURLException e) {
            failedUrlsList.add(absoluteUrl);
        }
        return new CheckerAsyncResult(Collections.singletonList(future), failedUrlsList, absoluteUrl);
    }


    private CheckerAsyncResult beginCheckRelativeUrl(String relativeUrl, RotorClientFactory rotorClientFactory) {
        List<String> failedUrlsList = new ArrayList<>();
        List<CompletableFuture<RedirectChain>> relativeFutures = new ArrayList<>();
        Boolean hasErrors = false;
        for (String protocol : Arrays.asList("http:", "https:")) {
            String newUrl = protocol + relativeUrl;
            try {
                relativeFutures.add(rotorClientFactory.getClient().fetchUrlTrace(new URL(newUrl)));
            } catch (MalformedURLException e) {
                hasErrors = true;
            }
            if (hasErrors) {
                failedUrlsList.add(relativeUrl);
            }
        }
        return new CheckerAsyncResult(relativeFutures, failedUrlsList, relativeUrl);
    }

     /*
         Страница разработана чтобы показывать в премодерации превью креативов для мобильного расхлопа, код которых в формате JSON.
         Мобильная морда вызывает авапс серверным вызовом, в котором получает json.
         Код используемый для показа взят с морды(Я).
         https://st.yandex-team.ru/BANNERSTORAGE-4563
    */
    @RequestMapping("preview_html5.jsp")
    public String preview_html5(@RequestParam("nmb") int nmb,
                                ModelMap model) throws SQLException {
        model.put("code", downloadFile(getCodeUrl(nmb)));
        return "creative/preview_html5";
    }

    /*
    Страница для тестирование шаблона-расхлоп для десктопной морды Я.
    https://st.yandex-team.ru/BANNERSTORAGE-4522
    */
    @RequestMapping("preview_yandex_main_page_rashlop_html5")
    public String preview_yandex_main_page_rashlop_html5(@RequestParam("nmb") int nmb,
                                                         ModelMap model) throws SQLException {
        model.put("code", downloadFile(getCodeUrl(nmb)));
        return "creative/preview_yandex_main_page_rashlop_html5";
    }

    @RequestMapping("preview_expandable_banner.jsp")
    public String preview_expandable_banner(@RequestParam("nmb") int nmb,
                                            ModelMap model) throws SQLException {
        model.put("code", downloadFile(getCodeUrl(nmb)));
        return "creative/preview_expandable_banner";
    }

    @RequestMapping("preview_expandable_banner_full.jsp")
    public String preview_expandable_banner_full(@RequestParam("nmb") int nmb,
                                            ModelMap model) throws SQLException {
        model.put("code", downloadFile(getCodeUrl(nmb)));
        return "creative/preview_expandable_banner_full";
    }

    @RequestMapping("preview_hidpi.jsp")
    public String preview_hidpi(@RequestParam("nmb") int nmb,
                                                 ModelMap model) throws SQLException {
        model.put("previewUrl", getCodeUrl(nmb));
        return "creative/preview_hidpi";
    }

    @RequestMapping("preview_tizer.jsp")
    public String preview_tizer(@RequestParam("nmb") int nmb,
            @RequestParam(value = "isRetina", defaultValue = "0") Boolean isRetina,
            ModelMap model) throws SQLException {
        model.put("code", downloadFile(getCodeUrl(nmb)));
        model.put("isRetina", isRetina);
        return "creative/preview_tizer";
    }

    @RequestMapping("landscape_preview.jsp")
    public String landscape_preview(@RequestParam("nmb") int nmb,
                                ModelMap model) throws SQLException {
        model.put("code", downloadFile(getCodeUrl(nmb)));
        return "creative/landscape_preview";
    }

    @RequestMapping("preview_retina_mobile.jsp")
    public String preview_retina_mobile(@RequestParam("nmb") int nmb,
                                        @RequestParam(value = "isRetina", defaultValue = "0") Boolean isRetina,
                                    ModelMap model) throws SQLException {
        model.put("code", downloadFile(getCodeUrl(nmb)));
        model.put("isRetina", isRetina);
        return "creative/preview_retina_mobile";
    }

    @RequestMapping("article_.jsp")
    public String article() {
        return "creative/article_";
    }

    @RequestMapping("brand_.jsp")
    public String brand() {
        return "creative/brand_";
    }

    private Map<String, List<Object>> getCreativeVersionParameters(int creativeVersionId) throws SQLException {
        Map<String, List<Object>> result = new HashMap<>();
        try (Connection connection = dataSource.getConnection()) {
            try (PreparedStatement preparedStatement = connection.prepareStatement(
                    "select\n" +
                    "    p.name,\n" +
                    "    value_int as int_value,\n" +
                    "    s.value as str_value\n" +
                    "from t_creative_version_param_value pv\n" +
                    "   join t_parameter p on pv.parameter_nmb = p.nmb\n" +
                    "   left join t_string s on pv.string_nmb = s.nmb\n" +
                    "where pv.creative_version_nmb = ?\n" +
                    "   and (value_int is not null or string_nmb is not null)\n" +
                    "order by p.name, order_nmb"
            )) {
                preparedStatement.setInt(1, creativeVersionId);
                try (ResultSet rs = preparedStatement.executeQuery()) {
                    while (rs.next()) {
                        String name = rs.getString("name");
                        Integer intValue = rs.getInt("int_value");
                        if (rs.wasNull()) {
                            intValue = null;
                        }
                        String strValue = rs.getString("str_value");
                        if (!result.containsKey(name)) {
                            result.put(name, new ArrayList<>());
                        }
                        result.get(name).add(ObjectUtils.firstNonNull(intValue, strValue));
                    }
                }
            }
        }
        return result;
    }

    @RequestMapping("index.jsp")
    public String viewIndex(@RequestParam(value = "nmb", defaultValue = "0") Integer creativeVersionNmb,
                            @RequestParam(value = "creative_nmb", defaultValue = "0") Integer nmb,
                            ModelMap model) throws SQLException {
        String geo = "";
        try (Connection connection = dataSource.getConnection()) {

            // передали не номер версии, а номер креатива. получение номера последней версии данного креатива
            if (creativeVersionNmb == 0 && nmb != 0) {
                try (PreparedStatement stmt = connection.prepareStatement(
                        "exec sp_moderation_creative_select 'get_last_version', @creative_nmb = ?")) {
                    stmt.setInt(1, nmb);
                    try (ResultSet rs = stmt.executeQuery()) {
                        if (rs.next()) creativeVersionNmb = rs.getInt("nmb");
                    }
                }
            }
            try (PreparedStatement stmt = connection.prepareStatement(
                    "exec sp_moderation_creative_select 'geo', @creative_version_nmb = ?")) {
                stmt.setInt(1, creativeVersionNmb);
                try (ResultSet rs = stmt.executeQuery()) {
                    while (rs.next()) {
                        String geoName = rs.getString("name");
                        int exclude = rs.getInt("exclude");
                        geo = geo + geoName + (exclude == 1 ? " (прореживание)" : "") + "; ";
                    }
                }
            }
        }
        model.addAttribute("geo", geo);
        model.addAttribute("sites", creativeService.getRedSites(creativeVersionNmb));
        model.addAttribute("red_pixel", creativeService.getRedPixel(creativeVersionNmb));
        model.addAttribute("urls", getCreativeUrls(creativeVersionNmb));
        model.addAttribute("creative_parameters", getCreativeVersionParameters(creativeVersionNmb));
        return "creative/index";
    }


    @RequestMapping("takeCreativeToModeration")
    public String takeCreativeToModeration(@RequestParam(value = "creative_tag", required = false) String creative_tag,
                                           @RequestParam(value = "creative_group_nmb", required = false) String creative_group_nmb,
                                           HttpServletRequest request,
                                           ModelMap model) throws SQLException {
        Integer error_code = 0;
        String result = "";

        try (Connection connection = dataSource.getConnection()) {
            try (PreparedStatement stmt = connection.prepareStatement(
                    "exec sp_moderation_creative_version_lock_by_tag @user_id = ?, @creative_tag = ?, @creative_group_nmb = ?")) {
                stmt.setString(1, ((Hashtable)request.getSession().getAttribute("user")).get("user_id").toString());
                stmt.setString(2, creative_tag);
                stmt.setString(3, creative_group_nmb);

                try (ResultSet rs = stmt.executeQuery()) {
                    while (rs.next()) {
                        error_code = rs.getInt("error_code");
                        result = rs.getString("result");
                    }
                }
            }
        }
        model.put("error_code", error_code);
        model.put("result", result);

        return "creative/msg_creative_moderation_lock";

    }

    @RequestMapping("show_banner.jsp")
    public String showBanner() {
        return "creative/show_banner";
    }

    @RequestMapping("template.jsp")
    public String template() {
        return "creative/template";
    }

    @RequestMapping("template_html5.jsp")
    public String template_html5(@RequestParam("nmb") int nmb,
                                 ModelMap model) throws SQLException {
        model.put("code", downloadFile(getCodeUrl(nmb)));
        model.put("creativeVersionNmb", nmb);
        return "creative/template_html5";
    }

    @RequestMapping("template_expandable_banner.jsp")
    public String template_expandable_banner(@RequestParam("nmb") int nmb,
                                             @RequestParam(value = "template_nmb", required = false) Integer templateNmb,
                                             ModelMap model) {
        model.put("creativeVersionNmb", nmb);
        String previewUrl;
        if (templateNmb == null) {templateNmb = 0;}
        if (templateNmb == 891) {
            previewUrl = "/creative/preview_expandable_banner_full.jsp?nmb=";
        } else {
            previewUrl = "/creative/preview_expandable_banner.jsp?nmb=";
        }
        model.put("previewUrl", previewUrl);
        return "creative/template_expandable_banner";
    }

    private String downloadFile(String url) {
        String code;
        try (CloseableHttpResponse response = httpClient.execute(new HttpGet(url))) {
            HttpEntity entity = response.getEntity();
            code = new String(ByteStreams.toByteArray(entity.getContent()), UTF_8);
            EntityUtils.consume(entity);
            return code;
        } catch (IOException e) {
            throw new UncheckedIOException(String.format("Can't read the file from url '%s'", url), e);
        }
    }

    @RequestMapping("template_list.jsp")
    public String templateList(@RequestParam("nmb") int nmb,
                               ModelMap model) throws SQLException {
        List<ParameterValueDTO> parameterValues = new ArrayList<>();
        try (Connection connection = dataSource.getConnection()) {
            try (PreparedStatement stmt = connection.prepareStatement(
                    "select * from [vt_creative_version_parameter] where creative_version_nmb = ?")) {
                stmt.setInt(1, nmb);

                try (ResultSet rs = stmt.executeQuery()) {
                    while (rs.next()) {
                        String name = rs.getString("parameter_name");
                        String value = rs.getString("value");
                        String parameterTypeName = rs.getString("parameter_type_name");

                        parameterValues.add(new ParameterValueDTO(name, value, parameterTypeName));
                    }
                }
            }
        }
        model.put("parameterValues", parameterValues);
        return "creative/template_list";
    }

    @RequestMapping(method = RequestMethod.GET, value = "xt_modify_creative.jsp")
    public String xtGetModifyCreative(@RequestParam("nmb") int creativeVersionId,
                                      HttpServletRequest request) throws SQLException {
        String redirect = "redirect:/index.jsp?status_nmb=2";
        String user_id = ((Hashtable) request.getSession().getAttribute("user")).get("user_id").toString();

        String action = request.getParameter("action");
        String all = request.getParameter("all");

        if (action.equals("clear")) {
            try (Connection connection = dataSource.getConnection()) {
                int creativeId;

                try (PreparedStatement stmt = connection.prepareStatement(
                        "select creative_nmb from t_creative_version where nmb = ?")) {
                    stmt.setInt(1, creativeVersionId);
                    try (ResultSet resultSet = stmt.executeQuery()) {
                        resultSet.next();
                        creativeId = resultSet.getInt("creative_nmb");
                    }
                }

                try (PreparedStatement stmt = connection.prepareStatement(
                        "exec sp_moderation_creative_version_clear @creative_version_nmb = ?, @user_id = ?")) {
                    stmt.setInt(1, creativeVersionId);
                    stmt.setString(2, user_id);
                    stmt.executeUpdate();
                    if (all == null) {
                        return "redirect:/index.jsp?creative_nmb=" + creativeId;
                    }
                }
            }
        }

        return redirect;
    }

    @RequestMapping(method = RequestMethod.POST, value="xt_modify_creative.jsp")
    public String xtPostModifyCreative(@RequestParam("nmb") String nmb,
                                       @RequestParam("action") String action,
                                       @RequestParam(value = "only_save_reason", required = false) @Nullable String onlySaveReason,
                                       @RequestParam(value = "send_email_to_ccd", required = false) @Nullable String sendEmailToCcd,
                                       @RequestParam(value = "remoderate", required = false) @Nullable String remoderate,
                                       @RequestParam(value = "res_brand", required = false) @Nullable String resBrand,
                                       @RequestParam(value = "res_article", required = false) @Nullable String resArticle,
                                       @RequestParam(value = "moderation_info", required = false) @Nullable String moderationInfo,
                                       @RequestParam(value = "wrong_moderation_result", required = false) @Nullable String wrongModerationResult,
                                       @RequestParam(value = "rejectDocument", required = false) @Nullable String[] rejectDocuments,
                                       HttpServletRequest request) throws SQLException {

        String redirect = "redirect:/index.jsp?status_nmb=2";
        String user_id = ((Hashtable)request.getSession().getAttribute("user")).get("user_id").toString();
        String reasons = "";

        Enumeration en = request.getParameterNames();
        while(en.hasMoreElements()) {
            String name = (String)en.nextElement();

            if(name.indexOf("reason_") == 0) {
                name = name.substring(7);
                String reason = name.substring(name.indexOf("_") + 1);
                String macros_nmb = name.substring(0, name.indexOf("_"));
                reasons = reasons + macros_nmb + "=" + reason + ";";
            }
        }

        if(moderationInfo == null) moderationInfo = "";


        //-----

        String rejectDocumentReasons = rejectDocuments != null ? String.join(";", rejectDocuments) + ";" : "";

        boolean b_send_email_to_ccd = false;
        if(sendEmailToCcd != null && sendEmailToCcd.equals("1")) b_send_email_to_ccd = true;

        String brands = resBrand != null ? Arrays.stream(resBrand.split("-")).filter(StringUtils::isNotEmpty).map(str -> "[" + str + "]").collect(joining(" ")) : "";

        int status_nmb = 4;
        if(onlySaveReason != null && onlySaveReason.equals("1")) status_nmb = 2;
        else if(reasons.length() > 0 || (rejectDocuments != null && rejectDocuments.length > 0)) status_nmb = 3;
        if(remoderate != null) status_nmb = Integer.parseInt(remoderate);

        //----------------------------------------------------
        //----------------------------------------------------
        if (action.equals("check_moderation_result")) {
            try (Connection connection = dataSource.getConnection()) {
                try (PreparedStatement stmt = connection.prepareStatement(
                        "exec sp_moderation_creative_version_check_moderation_result @creative_version_nmb = ?, @user_id = ?, @wrong_moderation_result = ?")) {
                    stmt.setString(1, nmb);
                    stmt.setString(2, user_id);
                    if(wrongModerationResult != null){ stmt.setString(3, "1"); }
                    else{ stmt.setString(3, "0  "); }
                    stmt.executeUpdate();
                    return "redirect:/creative/index.jsp?nmb=" + nmb;
                }
            }
        }

        //----------------------------------------------------
        //----------------------------------------------------
        if (action.equals("moderate")) {

            if(remoderate != null && !remoderate.isEmpty() && (remoderate.equals("2") || remoderate.equals("4"))){
                redirect = "redirect:/creative/index.jsp?nmb=" + nmb;
            }

            try (Connection connection = dataSource.getConnection()) {
                try (PreparedStatement stmt = connection.prepareStatement(
                        "exec sp_moderation_creative_article @mode = 'change', @creative_version_nmb = ?, @articles_str = ?")) {
                    stmt.setString(1, nmb);
                    stmt.setString(2, resArticle);
                    stmt.executeUpdate();
                }

                try (PreparedStatement stmt = connection.prepareStatement(
                        "exec sp_moderation_creative_brand @mode = 'change', @creative_version_nmb = ?, @brands_str = ?")) {
                    stmt.setString(1, nmb);
                    stmt.setString(2, brands);
                    stmt.executeUpdate();
                }

                try (CallableStatement stmt = connection.prepareCall(
                        "{call sp_creative_moderate_ (" +
                                "@creative_version_nmb = ?, " +
                                "@user_id = ?, " +
                                "@res_status_nmb = ?, " +
                                "@moderation_info = ?, " +
                                "@reasons = ?, " +
                                "@reject_document_reasons = ?, " +
                                "@send_email_to_ccd = ?, " +
                                "@ok = ?," +
                                "@res = ?)}")) {

                    stmt.setString(1, nmb);
                    stmt.setString(2, user_id);
                    stmt.setInt(3, status_nmb);
                    stmt.setString(4, moderationInfo);
                    stmt.setString(5, reasons);
                    stmt.setString(6, rejectDocumentReasons);
                    stmt.setString(7, (b_send_email_to_ccd ? "1" : "0"));
                    stmt.registerOutParameter(8, Types.INTEGER);
                    stmt.registerOutParameter(9, Types.VARCHAR);

                    String errCode = null;
                    String errorMessage = null;

                    stmt.execute();
                    try (ResultSet rs = stmt.getResultSet()){
                        if (rs == null) {
                            errCode = stmt.getString(8);
                            if (!errCode.isEmpty() && !errCode.equals("1")){
                                errorMessage = stmt.getString(9);
                            }

                            if((status_nmb == 2 && !b_send_email_to_ccd) || errorMessage != null) {
                                redirect = "redirect:/creative/index.jsp?nmb=" + nmb;
                                if (errorMessage != null) {
                                    try {
                                        redirect += "&moderate_error=" + URLEncoder.encode(errorMessage, StandardCharsets.UTF_8.name());
                                    } catch (UnsupportedEncodingException ignore) {
                                    }
                                }
                            }
                            return redirect;
                        }
                    } catch (SQLException e) {
                        throw new RuntimeException(e);
                    }

                    return redirect;
                }
            }
        }

       //-----

        return redirect;
    }

    @RequestMapping(value = "rejectReasons")
    @ResponseBody
    public ModerationResult getRejectReasons(@RequestParam("reason_object_id") String reasonObjectId,
                                             @RequestParam("creative_version_nmb") int creativeVersionNmb,
                                             @RequestParam("RUEN") String ruen,
                                             @RequestParam(value = "macrosNmb", required = false) Integer macrosNmb,
                                             @RequestParam(value = "parameterType", required = false) Integer parameterType) throws SQLException {
        if (macrosNmb != null && parameterType != null) throw new BadRequestException();
        List<RejectReasonDTO> rejectReasons = new ArrayList<>();
        if (reasonObjectId == null || reasonObjectId.isEmpty()) return new ModerationResult(rejectReasons, false);
        String sqlStatement = "exec sp_moderation_creative_reason 'select_for_object', @reason_object_id = ?, @creative_version_nmb = ?, @cmd = ?, @RUEN = ?";
        if (macrosNmb != null) {
            sqlStatement = sqlStatement + ", @macros_nmb = ?";
        }
        if (parameterType != null) {
            sqlStatement = sqlStatement + ", @parameter_type = ?";
        }
        boolean hasModerated;
        try (Connection conn = dataSource.getConnection()) {
            try (PreparedStatement stmt = conn.prepareStatement(sqlStatement)) {
                stmt.setString(1, reasonObjectId);
                stmt.setInt(2, creativeVersionNmb);
                stmt.setString(3, getCmd(creativeVersionNmb));
                stmt.setString(4, ruen);
                if (macrosNmb != null) stmt.setInt(5, macrosNmb);
                if (parameterType != null) stmt.setInt(5, parameterType);
                try (ResultSet rs = stmt.executeQuery()) {
                    rs.next();
                    hasModerated = rs.getBoolean("hasModerated");
                    if (stmt.getMoreResults()) {
                        ResultSet result = stmt.getResultSet();
                        while (result.next()) {
                            rejectReasons.add(new RejectReasonDTO(
                                    result.getString("id"),
                                    result.getInt("ok"),
                                    result.getInt("prev_ok"),
                                    result.getInt("nmb"),
                                    result.getString("moderator_name"),
                                    result.getInt("reason_type_nmb"),
                                    result.getString("reason_type_name"),
                                    result.getString("text")));
                        }
                    }
                }
            }
        }
        return new ModerationResult(rejectReasons, hasModerated);
    }

    @RequestMapping(value = "previewErrors")
    @ResponseBody
    public List<String> getCreativePreviewErrors(@RequestParam("creative_version_nmb") int creativeVersionNmb) throws SQLException {
        List<String> errors = new ArrayList<>();
        try (Connection conn = dataSource.getConnection()) {
            try (PreparedStatement stmt = conn.prepareStatement("SELECT error_message " +
                    " FROM dbo.t_creative_version_preview_error " +
                    " WHERE creative_version_nmb = ?" +
                    " ORDER BY error_message")) {
                stmt.setInt(1, creativeVersionNmb);
                try (ResultSet rs = stmt.executeQuery()) {
                    while (rs.next()) {
                        errors.add(rs.getString("error_message"));
                    }
                }
            }
        }
        return errors;
    }

    @RequestMapping(value = "getSites")
    @ResponseBody
    public List<SiteDTO> getSites(@RequestParam("creative_version_nmb") int creativeVersionNmb) throws SQLException {

        return creativeService.getAllSites(creativeVersionNmb);
    }

    @RequestMapping(value = "creativeDocuments")
    @ResponseBody
    public DocumentsDTO getCreativeDocuments(@RequestParam("creative_version_nmb") int creativeVersionNmb) throws SQLException {
        ArrayList<String> failedUrlsList = new ArrayList<>();
        ArrayList<String> noDocumentsDomains = new ArrayList<>();
        List<Map<String, String>> documents = new ArrayList<>();
        String domain = null;
        List<String> urls = getCreativeUrlsWithPixels(creativeVersionNmb);
        boolean isProtocolRelative;
        Map<String, CheckerAsyncResult> urlsCheckerMap = new HashMap<>();
        List<String> domains = new ArrayList<>();
        try (Connection connection = dataSource.getConnection()) {
            try (PreparedStatement stmt = connection.prepareStatement(
                    "select * from vt_creative_version_parameter where parameter_name in ('DOMAIN_LIST', 'PIXEL') and creative_version_nmb = ?"))
            {
                stmt.setInt(1, creativeVersionNmb);

                try (ResultSet rs = stmt.executeQuery()) {
                    while (rs.next()) {
                        domains.add(getUrlsDomain(new URL(rs.getString("value"))));

                    }
                } catch (MalformedURLException e) {
                    logger.error("Can't create url from value of DOMAIN_LIST parameter", e);
                }
            }
        }

        if (urls != null) {
            for (String url : urls) {
                isProtocolRelative = url.startsWith("//");
                if (isProtocolRelative) {
                    urlsCheckerMap.put(url, beginCheckRelativeUrl(url, rotorClientFactory));
                } else {
                    urlsCheckerMap.put(url, beginCheckAbsoluteUrl(url, rotorClientFactory));
                }
            }

            for (String url : urls) {
                CheckResult result = urlsCheckerMap.get(url).get();
                URL resultUrl = result.getResultUrl();
                List<String> failedUrls = result.getFailedUrls();

                if (resultUrl != null && !result.hasErrors()) {
                    domains.add(getUrlsDomain(resultUrl));
                }

                if (failedUrls.size() != 0) {
                    for (String failedUrl : failedUrls) {
                        failedUrlsList.add(failedUrl);
                    }
                }
            }

            try (Connection connection = dataSource.getConnection()) {
                ArrayList<String> documentIds = new ArrayList<>();
                // получаем список всех id документов для данного креатива
                try (PreparedStatement stmt = connection.prepareStatement(
                        "exec sp_document @mode ='select_for_creative', @creative_version_nmb = ?")) {
                    stmt.setInt(1, creativeVersionNmb);
                    try (ResultSet rs = stmt.executeQuery()) {
                        while (rs.next()) {
                            documentIds.add(rs.getString("id"));
                        }
                    }
                }
                // для каждого домена ищем соответствующий документ, не дублируя уже привязанные к креативу докменты
                for (String dmn : domains) {
                    try (PreparedStatement stmt = connection.prepareStatement(
                            "exec sp_document @domaine = ?")) {
                        stmt.setString(1, dmn);
                        try (ResultSet rs = stmt.executeQuery()) {
                            ResultSet result;
                            rs.next();
                            if (rs.getInt("cnt") != 0) {
                                if (stmt.getMoreResults()) {
                                    result = stmt.getResultSet();
                                    while (result.next()) {
                                        String document_id = result.getString("id");
                                        if (!documentIds.contains(document_id)) {
                                            documentIds.add(document_id);
                                        }
                                    }
                                }
                            } else {
                                noDocumentsDomains.add(dmn);
                            }
                        }
                    }
                }
                if (domains.size() != 0) {
                    domain = domains.get(0);
                }
                for (String id : documentIds) {
                    try (PreparedStatement stmt = connection.prepareStatement(
                            "exec sp_document @mode='select', @id = ?")) {
                        stmt.setString(1, id);
                        try (ResultSet rs = stmt.executeQuery()) {
                            while (rs.next()) {
                                Map<String, String> document = new HashMap<>();
                                document.put("document_id", rs.getString("id"));
                                document.put("document_type_name", rs.getString("document_type_name"));
                                document.put("document_description", rs.getString("description"));
                                document.put("expiration_date", rs.getString("expiration_date"));
                                document.put("expire", Integer.toString(rs.getInt("expire")));
                                documents.add(document);
                            }
                        }
                    }
                }
            }
        }
        return new DocumentsDTO(documents, failedUrlsList, noDocumentsDomains, domain);
    }

}
