package ru.yandex.controllers;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.ImmutableMap;
import org.apache.commons.lang3.StringUtils;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;
import org.springframework.jdbc.core.simple.SimpleJdbcInsert;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.servlet.view.RedirectView;
import ru.yandex.bannerstorage.premoderation.integration.stillage.StillageAnswer;
import ru.yandex.bannerstorage.premoderation.integration.stillage.StillageClient;

import javax.inject.Inject;
import javax.servlet.http.HttpServletRequest;
import java.sql.PreparedStatement;
import java.sql.Types;
import java.util.*;

/**
 * @author elwood
 */
@Controller
@RequestMapping("doc")
public class DocumentsController {

    private final ObjectMapper objectMapper = new ObjectMapper();

    private final JdbcTemplate jdbcTemplate;
    private final StillageClient stillageClient;

    private static final String CREATE_DOCUMENT_SQL = "sp_document_file @mode = 'create', @user_id = ?,"
            + " @document_id = ?, @file_instance_nmb = ?";


    @Inject
    public DocumentsController(final JdbcTemplate jdbcTemplate,
                               final StillageClient stillageClient) {
        this.jdbcTemplate = jdbcTemplate;
        this.stillageClient = stillageClient;
    }

    @RequestMapping("doc.jsp")
    public String doc() {
        return "doc/doc";
    }

    @RequestMapping("docs.jsp")
    public String docs() {
        return "doc/docs";
    }

    @RequestMapping("xt_delete.jsp")
    public String xtDelete() {
        return "doc/xt_delete";
    }

    @RequestMapping("xt_modify_creative_doc.jsp")
    public String xtModifyCreativeDoc() {
        return "doc/xt_modify_creative_doc";
    }

    @RequestMapping("xt_modify_doc.jsp")
    public String xtModifyDoc() {
        return "doc/xt_modify_doc";
    }


    @RequestMapping(value = "xt_modify_file.jsp", method = RequestMethod.POST)
    public RedirectView xtModifyFile(@RequestParam("file") final MultipartFile file,
                                     final HttpServletRequest request) throws Exception {

        final String document_id = request.getParameter("document_id");
        String fileName = request.getParameter("filename");

        if (fileName == null) {
            fileName = "";
        }

        if (fileName.contains("\\")) {
            fileName = StringUtils.substringAfterLast(fileName, "\\");
        }

        final String user_id = ((Hashtable) request.getSession().getAttribute("user")).get("user_id").toString();


        final StillageAnswer stillageAnswer = stillageClient.uploadFile(fileName, file.getBytes());

        final Integer mimeTypeNmb  = getMimeTypeNmb(stillageAnswer);

        final int fileNmb = getFileNmb(stillageAnswer, mimeTypeNmb);

        final int fileInstanceNmb = getFileInstanceNmb(fileNmb, -1, fileName);

        jdbcTemplate.update(connection -> {
            PreparedStatement ps = connection.prepareStatement(CREATE_DOCUMENT_SQL);
            ps.setString(1, user_id);
            ps.setString(2, document_id);
            ps.setInt(3, fileInstanceNmb);
            return ps;
        });

        return new RedirectView("/doc/doc.jsp?id=" + document_id);
    }


    private Integer getMimeTypeNmb(final StillageAnswer stillageAnswer) {
        if (stillageAnswer.getMimeType() != null) {
            final List<Integer> mimeTypeNmbs = jdbcTemplate
                    .queryForList("SELECT nmb FROM dbo.c_mime_type WHERE name = ?",
                            new Object[]{StringUtils.substringBefore(stillageAnswer.getMimeType(), "; charset=")},
                            new int[]{Types.NVARCHAR},
                            Integer.class);

            if (mimeTypeNmbs != null && !mimeTypeNmbs.isEmpty()) {
                return mimeTypeNmbs.iterator().next();
            } else {
                return null;
            }
        } else {
            return null;
        }
    }

    private int getFileNmb(final StillageAnswer stillageAnswer, final Integer mimeTypeNmb) throws JsonProcessingException {
        List<Integer> fileNmbs = jdbcTemplate.queryForList("SELECT nmb FROM dbo.t_file WHERE hash = ?",
                new Object[]{stillageAnswer.getMd5Hash()},
                new int[]{Types.BINARY},
                Integer.class);

        if (fileNmbs.isEmpty()) {

            final SimpleJdbcInsert jdbcInsertFile = new SimpleJdbcInsert(jdbcTemplate);
            jdbcInsertFile.withSchemaName("dbo")
                    .withTableName("t_file")
                    .usingColumns("hash", "width", "height", "stillage_file_id", "stillage_file_url", "stillage_file_metadata", "mime_type_nmb", "size")
                    .usingGeneratedKeyColumns("nmb");
            final Map<String, Object> parameters = new HashMap<>();
            parameters.put("hash", stillageAnswer.getMd5Hash());
            parameters.put("stillage_file_id", stillageAnswer.getId());
            parameters.put("stillage_file_url", stillageAnswer.getUrl());

            final Integer width, height;
            final String stillageFileMetadata;
            if (stillageAnswer.getMetadataInfo() != null) {
                final Map<String, Object> metadataInfo = stillageAnswer.getMetadataInfo();

                stillageFileMetadata = objectMapper.writeValueAsString(metadataInfo);
                width = (Integer) metadataInfo.get("width");
                height = (Integer) metadataInfo.get("height");
            } else {
                width = null;
                height = null;
                stillageFileMetadata = "{}";
            }
            parameters.put("width", width);
            parameters.put("height", height);
            parameters.put("stillage_file_metadata", stillageFileMetadata);
            parameters.put("mime_type_nmb", mimeTypeNmb);
            parameters.put("size", stillageAnswer.getFileSize());

            return jdbcInsertFile.executeAndReturnKey(new MapSqlParameterSource(parameters)).intValue();
        } else {
            return fileNmbs.iterator().next();
        }
    }

    private int getFileInstanceNmb(final Integer fileNmb, final int customerNmb, final String fileName) {
        Integer fileInstanceNmb = jdbcTemplate.queryForObject("SELECT MAX(nmb) " +
                        " FROM dbo.t_file_instance WHERE " +
                        " file_nmb = ? AND " +
                        " customer_nmb = ? AND " +
                        " [file_name] = ? AND " +
                        " [tag] = ?",
                Integer.class, fileNmb, customerNmb, fileName, "");
        if (fileInstanceNmb == null) {

            final SimpleJdbcInsert jdbcInsertFileInstance = new SimpleJdbcInsert(jdbcTemplate);
            jdbcInsertFileInstance.withSchemaName("dbo").withTableName("t_file_instance")
                    .usingColumns("file_nmb", "customer_nmb", "file_name", "tag", "file_added_date")
                    .usingGeneratedKeyColumns("nmb");

            fileInstanceNmb = jdbcInsertFileInstance.executeAndReturnKey(new MapSqlParameterSource(ImmutableMap.of(
                    "file_nmb", fileNmb,
                    "customer_nmb", customerNmb,
                    "file_name", fileName,
                    "tag", "",
                    "file_added_date", new Date()
            ))).intValue();
        }
        return fileInstanceNmb;
    }
}
