package ru.yandex.controllers;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import ru.yandex.JspUtils;

import javax.naming.NamingException;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import javax.sql.DataSource;
import java.io.IOException;
import java.sql.*;
import java.util.*;
import static com.google.common.base.MoreObjects.firstNonNull;

/**
 * @author elwood
 */
@Controller
public class IndexController {
    private static final Logger logger = LoggerFactory.getLogger(IndexController.class);

    private static final String START_PAGE = "/index.jsp?status_nmb=2";

    private final DataSource dataSource;

    @Autowired
    public IndexController(DataSource dataSource) {
        this.dataSource = dataSource;
    }

    @RequestMapping(path = {"", "index.jsp"})
    public String viewIndex() {
        return "index";
    }

    @RequestMapping(path = "partSelect", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseBody
    public ResponseEntity partSelect(@RequestParam("term") String term) throws SQLException {
        try (Connection db = dataSource.getConnection()) {
            String req = "exec sp_site_ @mode='select_like', @part = ?";

            try (PreparedStatement st = db.prepareStatement(req)) {
                st.setString(1, term);
                try (ResultSet rs = st.executeQuery()) {
                    ArrayList<Map<String, String>> sites = new ArrayList<>();
                    while (rs.next()) {
                        Map<String, String> suggestItems = new HashMap<>();
                        suggestItems.put("id", rs.getString("nmb"));
                        suggestItems.put("value", rs.getString("name"));
                        sites.add(suggestItems);
                    }
                    return ResponseEntity.status(HttpStatus.OK).body(sites);
                }
            }
        }
    }

    @RequestMapping(path = "Login")
    public String viewLogin() {
        return "Login";
    }

    @RequestMapping(method = RequestMethod.POST, path = "authorise")
    public String authorise(@RequestParam("login") String login,
                            @RequestParam("password") String password,
                            @RequestParam("RUEN") String ruEn,
                            HttpServletRequest request,
                            HttpServletResponse response) throws NamingException, SQLException, IOException {
        String returnUrl = firstNonNull((String) request.getSession().getAttribute("returnURL"), START_PAGE);

        try (Connection db = dataSource.getConnection()) {
            if (!JspUtils.isCorrect(login + password)) {
                return "redirect:/Login.jsp?message=authorisation_failed";
            }

            String user_id = "", full_name = "", msg = "", login_key = "";
            int ok = 0;
            Hashtable user = new Hashtable();

            String req = "exec sp_user_ 'authorise', @login = ?, @password = ?, @RUEN = ?";
            try (PreparedStatement st = db.prepareStatement(req)) {
                st.setString(1, login);
                st.setString(2, password);
                st.setString(3, ruEn);

                try (ResultSet rs = st.executeQuery()) {
                    if (rs.next()) {
                        user_id = rs.getString("id");
                        msg = rs.getString("msg");
                        ok = rs.getInt("ok");
                        full_name = rs.getString("full_name");
                        login_key = rs.getString("login_key");
                    }
                }

                if (ok == 0) {
                    return "redirect:/Login.jsp?msg=" + msg;
                }

                user.put("user_id", user_id);
                user.put("full_name", full_name);
                user.put("RUEN", ruEn);

                if (st.getMoreResults()) {
                    try (ResultSet rs = st.getResultSet()) {
                        while (rs.next()) {
                            String symbol = rs.getString("symbol");
                            user.put(symbol, "ok");
                        }
                    }
                }
            }
            request.getSession().setAttribute("user", user);

            // day login
            int end_hour = 23;

            java.util.Date dt = new java.util.Date();
            int h = dt.getHours();
            int m = dt.getMinutes();
            int s = dt.getSeconds();
            int max_age = (end_hour - h) * 3600 - m * 60 - s;

            Cookie ck = new Cookie("login_key", login_key);
            ck.setMaxAge(max_age);
            ck.setPath("/");
            response.addCookie(ck);

            return "redirect:" + returnUrl;
        }
    }

    @RequestMapping(path = "logout")
    public String logout(HttpServletRequest request, HttpServletResponse response) {
        HttpSession session = request.getSession();
        session.setAttribute("returnURL", null);
        session.setAttribute("user", null);

        // Reset autologin cookie
        response.addCookie(new Cookie("login_key", null));

        return "redirect:/Login.jsp";
    }

    @RequestMapping(path = "ping")
    @ResponseBody
    public ResponseEntity ping() {
        try (Connection connection = dataSource.getConnection();
             Statement statement = connection.createStatement()) {
            try (ResultSet resultSet = statement.executeQuery("select 1")) {
                if (resultSet.next() && resultSet.getInt(1) == 1) {
                    return ResponseEntity.status(HttpStatus.OK).body("0;OK");
                }
            }
        } catch (SQLException e) {
            logger.error("Test query failed", e);
        }
        return ResponseEntity.status(HttpStatus.SERVICE_UNAVAILABLE).body("2;Failed");
    }
}
