#!/bin/bash

set -e

if ! which realpath > /dev/null; then
    echo "To use $0 please install coreutils." >&2
    echo "  MacOS: brew install coreutils" >&2
    echo "         (see http://stackoverflow.com/questions/20381128/ if you don't have 'brew' installed)" >&2
    echo "  Linux: apt-get install coreutils" >&2
    exit 1;
fi

USAGE="Usage: $0 <dir> <package>. E.g.: $0 libs/my-new-library ru.yandex.direct.mynewlibrary"
DIR="${1?$USAGE}"
DIRNAME="$(basename "${DIR}")"
BASEDIR="$(dirname "${DIR}")"
YAMAKE="${BASEDIR}/ya.make"
NEW_YAMAKE="${BASEDIR}/ya.make-new"

PACKAGE="${2?$USAGE}"
PACKAGE_DIR="${PACKAGE//.//}"

ARCADIA_ROOT="${PWD}"
while [[ ! -e "${ARCADIA_ROOT}/.arcadia.root" ]]; do
    if [[ "${ARCADIA_ROOT}" == "/" ]]; then
        echo "$0: must be run from inside Arcadia checkout" >&2
        exit 1
    fi
    ARCADIA_ROOT="$(dirname "${ARCADIA_ROOT}")"
done

ARCDIR=$(realpath --relative-to="$ARCADIA_ROOT" "$DIR")

if [[ ! -e "${YAMAKE}" ]]; then
    echo "$0: ${YAMAKE} must exist." >&2
    exit 1
fi

if ! grep -q "RECURSE(" "${YAMAKE}"; then
    echo "$0: ${YAMAKE} must contain RECURSE() section." >&2
    exit 1
fi

mkdir "${DIR}" # Не перезаписываем существующий модуль

trap "rm -rf ${DIR} ${NEW_YAMAKE}" ERR

mkdir -p "${DIR}/src/main/java/${PACKAGE_DIR}"
mkdir -p "${DIR}/src/main/resources/"
mkdir -p "${DIR}/src/test/java/${PACKAGE_DIR}"
mkdir -p "${DIR}/src/test/resources/"
mkdir -p "${DIR}/ut"

echo placeholder > "${DIR}/src/main/java/${PACKAGE_DIR}/.placeholder"
echo placeholder > "${DIR}/src/main/resources/.placeholder"
echo placeholder > "${DIR}/src/test/java/${PACKAGE_DIR}/.placeholder"
echo placeholder > "${DIR}/src/test/resources/.placeholder"

echo -e "# ${DIRNAME} - что это\n\n### Подзаголовок\nтекст\n" >"${DIR}/README.md"

cat >"${DIR}/ya.make" <<EOF
JAVA_LIBRARY()
WITH_KOTLIN()
INCLUDE(\${ARCADIA_ROOT}/direct/libs/common.inc)

OWNER(g:direct)

PEERDIR(
)

JAVA_SRCS(SRCDIR src/main/java **/*)
JAVA_SRCS(SRCDIR src/main/resources **/*)

END()

RECURSE_FOR_TESTS(ut)
EOF

cat >"${DIR}/ut/ya.make" <<EOF
JUNIT5()
WITH_KOTLIN()
INCLUDE(\${ARCADIA_ROOT}/direct/libs/common_junit_ut.inc)

OWNER(g:direct)

JAVA_SRCS(SRCDIR \${ARCADIA_ROOT}/${ARCDIR}/src/test/java **/*)
JAVA_SRCS(SRCDIR \${ARCADIA_ROOT}/${ARCDIR}/src/test/resources **/*)

PEERDIR(
    ${ARCDIR}
)

END()
EOF

SRC="${YAMAKE}" DIRNAME="${DIRNAME}" awk 'BEGIN{
    inserted = 0;
}
END {
    if (!inserted) {
        print "WARNING: Failed to insert new library into RECURSE section of " ENVIRON["SRC"] ". Pls do it yourself." >> "/dev/stderr"
    }
}
{
    if ($0 == "RECURSE(") {
        print; getline;
        while ($1 < ENVIRON["DIRNAME"] && !index($0, ")")) {
            print; getline;
        }
        print "    " ENVIRON["DIRNAME"];
        inserted = 1;
        while (!index($0, ")")) {
            print; getline;
        }
        print;
    } else {
        print;
    }
}' "${YAMAKE}" > "${NEW_YAMAKE}"
mv "${NEW_YAMAKE}" "${YAMAKE}"

