#!/usr/bin/env python
# coding=utf-8

import yt.wrapper as yt
import argparse


def restore_table(src, dst):
    if yt.exists(dst):
        yt.remove(dst)
    yt.create_table(dst, attributes={
        'optimize_for': 'scan',
        'schema': yt.get_attribute(src, 'schema'),
        'primary_medium': 'ssd_blobs'
    })
    # Делаем блоки меньшего размера (см https://wiki.yandex-team.ru/yt/userdoc/dynamictablesmapreduce/)
    yt.run_merge(src, dst, mode='sorted', spec={
        'merge_by': [str(s) for s in yt.get_attribute(src, 'key_columns')],
        'job_io': {'table_writer': {'block_size': 256 * 2 ** 10, 'desired_chunk_size': 100 * 2 ** 20}},
        'force_transform': True
    })
    yt.alter_table(dst, dynamic=True)
    yt.mount_table(dst)

def store_table(src, dst):
    if yt.exists(dst):
        yt.remove(dst)
    yt.create_table(dst, attributes={
        'optimize_for': 'scan',
        'schema': yt.get_attribute(src, 'schema'),
        'primary_medium': 'ssd_blobs'
    })
    # Делаем блоки большего размера (см https://wiki.yandex-team.ru/yt/userdoc/dynamictablesmapreduce/)
    yt.run_merge(src, dst, mode='ordered', spec={
        'merge_by': [str(s) for s in yt.get_attribute(src, 'key_columns')],
        'job_io': {'table_writer': {'block_size': 10240 * 2 ** 10, 'desired_chunk_size': 300 * 2 ** 20}},
        'force_transform': True
    })

def restore_dir(args):
    """
        Для указанной в args.base_dir директории выполняет восстановление таблиц синхронизатора
        Статические таблицы преобразуются в динамические с правильным размером блока
        На map_node, соответствующей base_dir, устанавливается атрибут @initial-import-finished = %true
        Статические таблицы при этом не удаляются
    """
    base_dir = args.base_dir
    static_tables = [base_dir + '/mysql-sync-states_static'] + [base_dir + '/combined/' + table for table in
                                                                yt.list(base_dir + '/combined')]

    # Восстановим значение атрибута @initial-import-finished
    # По значению этого атрибута синхронизатор понимает, что начальный импорт завершён
    yt.set_attribute(base_dir, 'initial-import-finished', True)

    for static_table in static_tables:
        if not static_table.endswith('_static'):
            print('Skip {0}'.format(static_table))
            continue
        dynamic_table = static_table[:-len('_static')]
        restore_table(static_table, dynamic_table)

    print('Directory restored OK')

def store_dir(args):
    """
        Для указанной в args.base_dir директории выполняет дамп таблиц синхронизатора
        Динамические таблицы преобразуются в статические с правильным размером блока
        На map_node, соответствующей base_dir, устанавливается атрибут @initial-import-finished = %true
        Динамические таблицы при этом не удаляются
    """
    base_dir = args.base_dir
    dynamic_tables = [base_dir + '/mysql-sync-states'] + [base_dir + '/combined/' + table for table in
                                                                yt.list(base_dir + '/combined')]

    # Восстановим значение атрибута @initial-import-finished
    # По значению этого атрибута синхронизатор понимает, что начальный импорт завершён
    yt.set_attribute(base_dir, 'initial-import-finished', True)

    for dynamic_table in dynamic_tables:
        store_table(dynamic_table, dynamic_table + '_static')
    print('Directory stored OK')


def main():
    parser = argparse.ArgumentParser(description='Mysql→YT sync backup restore tool. Use YT_PROXY variable to set the YT cluster.')
    subparsers = parser.add_subparsers(help='Commands')

    parser_restore = subparsers.add_parser('restore')
    parser_store = subparsers.add_parser('store')
    parser_restore.add_argument('base_dir', type=str, help='Path to directory with backup data (static tables)')
    parser_restore.set_defaults(func=restore_dir)
    parser_store.add_argument('base_dir', type=str, help='Path to directory with data (dynamic tables)')
    parser_store.set_defaults(func=store_dir)


    # TODO :
    #   добавить команду cleanup, которая бы удаляла статические таблицы
    #   Можно ещё добавить флаг --cleanup в restore

    args = parser.parse_args()
    args.func(args)


if __name__ == "__main__":
    main()
