#!/usr/bin/python
# -*- coding: utf-8 -*-

import sys
sys.path.insert(0, '/opt/direct-py/startrek-python-client-sni-fix')

from argparse import ArgumentParser, RawTextHelpFormatter
from datetime import datetime, timedelta
from itertools import cycle
import json
import os
import re
from time import sleep

import requests
import pytz


SB_API = 'https://sandbox.yandex-team.ru/api/v1.0'

headers = {}
if 'SANDBOX_TOKEN' in os.environ:
    headers['Authorization'] = 'OAuth %s' % os.environ['SANDBOX_TOKEN']

class SB_STATUS:
    SUCCESS = 'SUCCESS'
    ENQUEUING = 'ENQUEUING'
    ENQUEUED = 'ENQUEUED'
    ASSIGNED = 'ASSIGNED'
    PREPARING = 'PREPARING'
    TEMPORARY = 'TEMPORARY'
    EXECUTING = 'EXECUTING'
    FINISHING = 'FINISHING'
    FAILURE = 'FAILURE'
    EXCEPTION = 'EXCEPTION'
    TIMEOUT = 'TIMEOUT'
    EXPIRED = 'EXPIRED'
    STOPPING = 'STOPPING'
    STOPPED = 'STOPPED'
    SUSPENDING = 'SUSPENDING'
    SUSPENDED = 'SUSPENDED'

    @staticmethod
    def JOB_FINISHED_STATUSES():
        return {SB_STATUS.SUCCESS, SB_STATUS.FAILURE, SB_STATUS.EXCEPTION, SB_STATUS.TIMEOUT, SB_STATUS.EXPIRED, SB_STATUS.STOPPING, SB_STATUS.STOPPED, SB_STATUS.SUSPENDING, SB_STATUS.SUSPENDED}

MoscowTZ = pytz.timezone('Europe/Moscow')

loads_time = lambda x: datetime.strptime(x, '%Y-%m-%dT%H:%M:%S.%fZ')
def dumps_time(x):
    if type(x) == unicode or type(x) == str:
        return dumps_time(loads_time(x))
    else:
        return MoscowTZ.localize(x).strftime('%Y-%m-%d %H:%M:%S')


def parse_arguments():
    parser = ArgumentParser(prog='sandbox-cli', formatter_class=RawTextHelpFormatter)
    subparsers = parser.add_subparsers(help=u'одна из команд из списка', metavar='COMMAND', dest='command')

    def set_common_arguments(parser):
        parser.add_argument( '--owner', '-o', dest='owner', required=False, type=str, help='sandbox owner')
        parser.add_argument( '--task-type', '-t', dest='task_type', required=True, type=str, help='sandbox task type')
        parser.add_argument( '--input-params', '-i', dest='input_params', required=False, type=str, help='''sandbox task input parameters
    example:
        key1=value1,key2=value2''')
        parser.add_argument('--tags', '-T', dest='tags', required=False, type=str, help='''sandbox task tags
    example:
        TAG1,TAG2,TAG3''')
        parser.add_argument('--hints', '-H', dest='hints', required=False, type=str, help='''sandbox hints
    example:
        HINT1,HINT2,HINT3''')

    locate_subparser = subparsers.add_parser('locate', help=u'отыскать тасочку')
    set_common_arguments(locate_subparser)
    locate_subparser.set_defaults(func=locate_cmd)

    await_subparser = subparsers.add_parser('await', help=u'подождать тасочку')
    set_common_arguments(await_subparser)
    await_subparser.set_defaults(func=await_cmd)

    kill_subparser = subparsers.add_parser('kill', help=u'прибить тасочку')
    set_common_arguments(kill_subparser)
    kill_subparser.set_defaults(func=kill_cmd)

    download_subparser = subparsers.add_parser('download-artifact', help=u'скачать артифакт')
    set_common_arguments(download_subparser)
    download_subparser.add_argument( '--artifact-type', '-a', dest='artifact_type', required=True, type=str, help='sandbox artifact type')
    download_subparser.add_argument( '--output', '-O', dest='output_filename', required=False, type=str, help='artifact archive filename', default='build.tgz')
    download_subparser.set_defaults(func=download_artifact)

    args = parser.parse_args()

    if args.input_params:
        args.input_params = { m.group(1): m.group(2) for m in [re.match('([^=]*)=(.*)', p) for p in args.input_params.split(',')] }

    return args

def download_file(url, output):
    i = 0
    chunk_size_kb = 40
    with requests.get(url, stream=True) as r:
        r.raise_for_status()
        with open(output, 'wb') as f:
            for chunk in r.iter_content(chunk_size=chunk_size_kb * 1024): 
                # If you have chunk encoded response uncomment if
                # and set chunk_size parameter to None.
                #if chunk: 
                f.write(chunk)
                i += chunk_size_kb
                print '\r%s kb' % i,
                sys.stdout.flush()

def locate_task(args):
    task_params = {
        'type': args.task_type,
        'limit': 1,
        'offset': 0,
        'order': '-id',
    }
    if args.owner:
        task_params['owner'] = args.owner
    if args.tags:
        task_params['tags'] = args.tags.split(',')
        task_params['all_tags'] = True
    if args.hints:
        task_params['hints'] = args.hints.split(',')
    if args.input_params:
        task_params['input_parameters'] = json.dumps(args.input_params)

    tasks = requests.get('%s/task' % SB_API, headers=headers, params=task_params).json()

    try:
        task = tasks['items'][0]
    except IndexError:
        raise Exception("Task with specified attributes not found!")

    task_id = task['id']
    task_status = task['status']
    print('Located task https://sandbox.yandex-team.ru/task/%s <%s>' % (task_id, task_status))

    return task


def await_task(task_id):
    is_job_finished = lambda x: x['status'] in SB_STATUS.JOB_FINISHED_STATUSES()
    spinner = cycle('\|/-')

    task_declared = False

    while True:
        task = requests.get('%s/task/%s' % (SB_API, task_id), headers=headers).json()

        if is_job_finished(task):
            break

        if not task_declared:
            task_declared = True
            task_started = dumps_time(task['time']['created'])
            print('task started at %s' % task_started)
            print('awaiting...')

        range_size = 150
        sleep_time = 15. / range_size
        for _ in range(range_size):
            sleep(sleep_time)
            print '\r%s' % next(spinner),
            sys.stdout.flush()

    print('\r')

    task_status = task['status']
    if task_status != SB_STATUS.SUCCESS:
        raise Exception('Task finished with %s status!' % task_status)

def locate_cmd(args):
    try:
        return locate_task(args)['id']
    except:
        return None

def await_cmd(args):
    task = locate_task(args)
    await_task(task['id'])

def kill_cmd(args):
    task = None
    try:
        task = locate_task(args)
    except:
        return
    print requests.put('%s/batch/tasks/delete' % SB_API, headers=headers, json={
        "id": [task['id']]
    }).text

def download_artifact(args):
    task_id = locate_task(args)['id']
    await_task(task_id)

    print('downloading artifact...')

    resource_params = {
        'task_id': task_id,
        'type': args.artifact_type,
        'limit': 1,
    }
    resources = requests.get('%s/resource' % SB_API, headers=headers, params=resource_params).json()

    resource_proxy = resources['items'][0]['http']['proxy']
    download_file('%s?stream=tgz' % resource_proxy, args.output_filename)

    print('artifact downloaded to %s' % args.output_filename)


if __name__ == '__main__':
    args = parse_arguments()
    args.func(args)
