package ru.yandex.bayan2;

import java.util.ArrayList;
import java.util.List;

import static java.util.stream.Collectors.joining;

/**
 * Инструмент для кодирования списка строк в одной строке.
 * Выходная строка не содержит переводов строк (они эскейпятся).
 *
 * @author elwood
 */
public class CombineSplitUtil {
    /**
     * Кодирует список строк в одной строке. Разделитель - «|». В исходных строках
     * этот символ экранируется обратным слешем, а обратный слеш - дублируется.
     * Дополнительно перевод строки заменяется на \n.
     * Если крайний элемент списка - пустая строка, то в конец результата добавляется ещё один символ «|».
     * Это позволяет отличать, к примеру, пустой список от списка, в котором единственный эл-т - пустая строка.
     */
    public static String combine(List<String> strings) {
        if (null == strings) {
            throw new IllegalArgumentException("strings shouldn't be null");
        }
        final String joined = strings.stream()
                .map(s -> s.replace("\\", "\\\\"))
                .map(s -> s.replace("|", "\\|"))
                .map(s -> s.replace("\n", "\\n"))
                .collect(joining("|"));
        return !strings.isEmpty() && strings.get(strings.size() - 1).equals("") ? joined + "|" : joined;
    }

    /**
     * Выполняет обратное преобразование из строки в список.
     */
    public static List<String> split(String combined) {
        if (null == combined) {
            throw new IllegalArgumentException("combined shouldn't be null");
        }
        List<String> list = new ArrayList<>();
        StringBuilder current = null;
        boolean waitingSuffix = false;
        for (int i = 0, len = combined.length(); i < len; i++) {
            final char c = combined.charAt(i);
            if (waitingSuffix) {
                if (current == null) {
                    current = new StringBuilder();
                }
                //
                if (c == '\\') {
                    current.append('\\');
                } else if (c == '|') {
                    current.append('|');
                } else if (c == 'n') {
                    current.append("\n");
                } else {
                    throw new IllegalArgumentException("Bad format");
                }
                waitingSuffix = false;
            } else {
                if (c == '\\') {
                    waitingSuffix = true;
                } else if (c == '|') {
                    list.add(current != null ? current.toString() : "");
                    current = null;
                } else {
                    if (null == current) {
                        current = new StringBuilder();
                    }
                    current.append(c);
                }
            }
        }
        if (null != current) {
            list.add(current.toString());
        }
        return list;
    }
}
