package ru.yandex.canvas.configs;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.handler.HandlerInterceptorAdapter;

import ru.yandex.canvas.configs.auth.AuthorizeBy;
import ru.yandex.canvas.configs.auth.CanvasAuthorizer;
import ru.yandex.canvas.exceptions.AuthException;
import ru.yandex.canvas.exceptions.UnauthorizedException;

import static ru.yandex.canvas.configs.auth.AuthorizeBy.AuthType.BLACKBOX;

/**
 * <p>
 * Специальный перехватчик запросов, который добавляет к ним авторизацию.
 * Создается через специальный билдер, который регистрирует модули которые умеют разные виды авторизации.
 * Там же указывается тип авторизации по умолчанию.
 * Контроллеры могут быть помечены аннотацией @AuthorizeBy, если ее нет, то используется авторизация по умолчанию.
 * </p>
 *
 * @see HandlerInterceptorAdapter
 * @see CanvasAuthorizer
 * @see AuthorizeBy
 */

public class CanvasAuthInterceptor extends HandlerInterceptorAdapter {
    private static final Logger logger = LoggerFactory.getLogger(CanvasAuthInterceptor.class);


    private Map<AuthorizeBy.AuthType, CanvasAuthorizer> authorizers;
    private List<AuthorizeBy.AuthType> defaultTypes;

    private CanvasAuthInterceptor(
            Map<AuthorizeBy.AuthType, CanvasAuthorizer> authorizers, List<AuthorizeBy.AuthType> defaultTypes)
    {
        this.authorizers = authorizers;
        this.defaultTypes = defaultTypes;

        if (defaultTypes == null) {
            defaultTypes = Collections.emptyList();
        }

        if (!defaultTypes.stream().allMatch(authorizers::containsKey)) {
            throw new IllegalArgumentException("Authorizers don't contain provided defaultTypes: " + defaultTypes);
        }

    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {
        private Map<AuthorizeBy.AuthType, CanvasAuthorizer> authorizers;
        private List<AuthorizeBy.AuthType> defaultType = List.of(BLACKBOX);

        private Builder() {
            this.authorizers = new HashMap<>();
        }

        public Builder defaultAuth(List<AuthorizeBy.AuthType> authTypes) {
            this.defaultType = authTypes;
            return this;
        }

        public Builder register(AuthorizeBy.AuthType authType, CanvasAuthorizer authorizer) {
            Objects.requireNonNull(authType);
            Objects.requireNonNull(authorizer);

            authorizers.put(authType, authorizer);
            return this;
        }

        public CanvasAuthInterceptor build() {
            return new CanvasAuthInterceptor(Collections.unmodifiableMap(authorizers), defaultType);
        }
    }

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler)
            throws UnauthorizedException, AuthException
    {
        if (!(handler instanceof HandlerMethod)) {
            return true;
        }

        HandlerMethod handlerMethod = (HandlerMethod) handler;

        if (handlerMethod.getMethod().getName().equals("error")) {
            return true;
        }

        AuthorizeBy authorizeByAnnotation = handlerMethod.getMethodAnnotation(AuthorizeBy.class);

        if (authorizeByAnnotation == null) {
            authorizeByAnnotation = handlerMethod.getBeanType().getAnnotation(AuthorizeBy.class);
        }

        if (authorizeByAnnotation != null && authorizeByAnnotation.value().length == 0) {
            throw new InternalError("AuthorizeBy annotation has incorrect (empty) value!");
        }

        Iterable<AuthorizeBy.AuthType> authTypes;

        if (authorizeByAnnotation != null && authorizeByAnnotation.value().length > 0) {
            authTypes = Arrays.asList(authorizeByAnnotation.value());
        } else {
            authTypes = defaultTypes;
        }

        RuntimeException exception = null;

        for (AuthorizeBy.AuthType authType : authTypes) {
            String callerName =
                    ((HandlerMethod) handler).getBean().getClass().getCanonicalName() + "." + handlerMethod.getMethod()
                            .getName();
            logger.info(callerName + ": selected auth method: " + authType);
            CanvasAuthorizer canvasAuthorizer = authorizers.get(authType);

            if (canvasAuthorizer == null) {
                logger.warn("Authorizers don't contain value from AuthorizeBy annotation: " + authType);
                continue;
            }

            try {
                canvasAuthorizer.authorize(request, authorizeByAnnotation);
                return true;
            } catch (RuntimeException e) {
                exception = e;
            }

        }

        if (exception == null) {
            throw new AuthException("Auth failed");
        }

        //Ситуация: перебрали все authTypes, авторизоваться не смогли. Где-то было исключение
        logger.warn("Auth failed with exception: " + exception.getMessage());
        //обернули в наше исключение чтобы отвечать 403, а не 500
        //no valid oauth credentials provided логичнее отдавать как 403
        throw new AuthException(exception.getMessage());
    }

}
