package ru.yandex.canvas.configs;

import java.util.Collection;
import java.util.Collections;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.servlet.http.HttpServletResponse;
import javax.validation.ConstraintViolation;
import javax.validation.ConstraintViolationException;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.ImmutableMap;
import com.google.gson.Gson;
import org.eclipse.jetty.http.BadMessageException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.stereotype.Component;
import org.springframework.validation.BindException;
import org.springframework.validation.FieldError;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import ru.yandex.canvas.exceptions.SandboxConvertionException;
import ru.yandex.canvas.exceptions.SourceValidationError;
import ru.yandex.canvas.exceptions.ValidationResultRepresentationException;

@ControllerAdvice
@Component
public class GlobalExceptionHandler {
    final Logger logger = LoggerFactory.getLogger(GlobalExceptionHandler.class);

    @ExceptionHandler({MethodArgumentNotValidException.class, HttpMessageNotReadableException.class,
            BindException.class})
    @ResponseBody
    @ResponseStatus(HttpStatus.BAD_REQUEST)
    public ResponseEntity<?> handle(MethodArgumentNotValidException exception) {
        return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(
                getMessage(exception.getBindingResult().getFieldErrors(), FieldError::getDefaultMessage)
        );
    }


    @ExceptionHandler(ConstraintViolationException.class)
    @ResponseBody
    @ResponseStatus(HttpStatus.BAD_REQUEST)
    public ResponseEntity<?> handle(ConstraintViolationException exception) {
        return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(
                getMessage(exception.getConstraintViolations(), ConstraintViolation::getMessage)
        );
    }

    @ExceptionHandler(SourceValidationError.class)
    @ResponseBody
    public ResponseEntity<?> handle(SourceValidationError exception, HttpServletResponse response) {
        return ResponseEntity
                .status(HttpStatus.BAD_REQUEST)
                .contentType(MediaType.APPLICATION_JSON)
                .body(exception.getMessage());
    }

    @ExceptionHandler(SandboxConvertionException.class)
    @ResponseBody
    public ResponseEntity<?> handle(SandboxConvertionException exception, HttpServletResponse response) {
        return ResponseEntity
                .status(HttpStatus.INTERNAL_SERVER_ERROR)
                .contentType(MediaType.APPLICATION_JSON)
                .body(new Gson().toJson(ImmutableMap
                        .of("message", Collections.singletonList(exception.getMessage()), "file_id",
                                exception.getFileId())));
    }


    @ExceptionHandler(ValidationResultRepresentationException.class)
    @ResponseBody
    public ResponseEntity<?> handle(ValidationResultRepresentationException exception) {
        try {
            String errorMessage = new ObjectMapper().writeValueAsString(exception.getValidationErrorRepresentation());
            logger.warn("Validation failed: {}", errorMessage);
        } catch (JsonProcessingException e) {
            logger.warn("Exception while processing error", e);
        }

        return ResponseEntity
                .status(HttpStatus.BAD_REQUEST)
                .contentType(MediaType.APPLICATION_JSON)
                .body(exception.getValidationErrorRepresentation());
    }

    @ExceptionHandler(BadMessageException.class)
    @ResponseBody
    public ResponseEntity<?> handle(BadMessageException exception, HttpServletResponse response) {

        return ResponseEntity
                .status(HttpStatus.PAYLOAD_TOO_LARGE)
                .contentType(MediaType.APPLICATION_JSON)
                .body(exception.getMessage());
    }

    private <T> String getMessage(Collection<T> collection, Function<T, String> mapper) {
        return collection.stream().map(mapper).distinct().sorted().collect(Collectors.joining(", "));
    }

}
