package ru.yandex.canvas.configs;

import java.util.Arrays;

import org.bson.internal.Base64;
import org.bson.types.Binary;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.convert.converter.Converter;
import org.springframework.core.convert.converter.ConverterFactory;
import org.springframework.data.convert.ReadingConverter;
import org.springframework.data.convert.WritingConverter;
import org.springframework.data.mongodb.core.convert.MongoCustomConversions;

import ru.yandex.canvas.MongoLowerCaseEnum;

@Configuration
public class MongoCustomConversionsConfig {

    @Bean
    public MongoCustomConversions customConversions() {
        return new MongoCustomConversions(
                Arrays.asList(new StringToEnumConverterFactory(), new EnumToLowercaseStringConverter(),
                        new BinaryToStringConverter())
        );
    }

    @ReadingConverter
    private class BinaryToStringConverter implements Converter<Binary, String> {
        @Override
        public String convert(Binary source) {
            return Base64.encode(source.getData());
        }
    }

    @ReadingConverter
    private static class StringToEnumConverterFactory implements ConverterFactory<String, Enum> {

        @Override
        public <T extends Enum> Converter<String, T> getConverter(Class<T> targetType) {
            return new StringToEnum(getEnumType(targetType));
        }

        @ReadingConverter
        private static class StringToEnum<T extends Enum> implements Converter<String, T> {

            private final Class<T> enumType;

            public StringToEnum(Class<T> enumType) {
                this.enumType = enumType;
            }

            @Override
            public T convert(String source) {
                if (source.isEmpty()) {
                    // It's an empty enum identifier: reset the enum value to null.
                    return null;
                }

                for (T t : enumType.getEnumConstants()) {
                    if (t instanceof MongoLowerCaseEnum) {
                        if (t.name().toLowerCase().equals(source)) {
                            return t;
                        }
                    }
                }

                return (T) Enum.valueOf(this.enumType, source.trim());
            }
        }

        static Class<?> getEnumType(Class<?> targetType) {
            Class<?> enumType = targetType;
            while (enumType != null && !enumType.isEnum()) {
                enumType = enumType.getSuperclass();
            }
            if (enumType == null) {
                throw new IllegalArgumentException(
                        "The target type " + targetType.getName() + " does not refer to an enum");
            }
            return enumType;
        }
    }

    @WritingConverter
    private static class EnumToLowercaseStringConverter implements Converter<Enum<?>, String> {
        @Override
        public String convert(Enum<?> source) {
            if (source instanceof MongoLowerCaseEnum) {
                return source.name().toLowerCase();
            } else {
                return source.name();
            }
        }
    }

}
