package ru.yandex.canvas.configs;

import java.io.IOException;
import java.io.InputStreamReader;
import java.net.MalformedURLException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Executor;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.servlet.http.HttpServletRequest;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.datatype.jdk8.Jdk8Module;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import org.apache.http.client.HttpClient;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.DefaultAsyncHttpClient;
import org.asynchttpclient.DefaultAsyncHttpClientConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.context.annotation.Lazy;
import org.springframework.context.annotation.Primary;
import org.springframework.context.support.PropertySourcesPlaceholderConfigurer;
import org.springframework.core.env.ConfigurableEnvironment;
import org.springframework.core.env.MutablePropertySources;
import org.springframework.core.env.PropertySource;
import org.springframework.data.mongodb.core.MongoOperations;
import org.springframework.http.client.BufferingClientHttpRequestFactory;
import org.springframework.http.client.ClientHttpRequestInterceptor;
import org.springframework.http.client.ClientHttpResponse;
import org.springframework.http.client.SimpleClientHttpRequestFactory;
import org.springframework.http.converter.protobuf.ProtobufHttpMessageConverter;
import org.springframework.http.converter.xml.MappingJackson2XmlHttpMessageConverter;
import org.springframework.scheduling.TaskScheduler;
import org.springframework.scheduling.concurrent.ThreadPoolTaskScheduler;
import org.springframework.util.CollectionUtils;
import org.springframework.util.FileCopyUtils;
import org.springframework.web.client.DefaultResponseErrorHandler;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.context.annotation.RequestScope;

import ru.yandex.canvas.RestTemplateLoggerInterceptor;
import ru.yandex.canvas.configs.auth.AuthorizeBy;
import ru.yandex.canvas.configs.auth.BlackBoxAuthorizer;
import ru.yandex.canvas.configs.auth.CanvasAuthorizer;
import ru.yandex.canvas.configs.auth.DirectTokenAuthorizer;
import ru.yandex.canvas.configs.auth.OAuthAuthorizer;
import ru.yandex.canvas.configs.auth.QueryStringAuthorizer;
import ru.yandex.canvas.configs.auth.SandboxCallbackAuthorizer;
import ru.yandex.canvas.configs.auth.SandboxCleanupHandleAuthorizer;
import ru.yandex.canvas.configs.auth.TvmAuthorizer;
import ru.yandex.canvas.controllers.video.PythonRedirect;
import ru.yandex.canvas.model.html_builder.CreativeHtmlPreviewWriter;
import ru.yandex.canvas.model.html_builder.CreativeHtmlRtbHostWriter;
import ru.yandex.canvas.model.html_builder.GeoPinCreativeHtmlWriter;
import ru.yandex.canvas.model.html_builder.ImageGeoPinCreativeHtmlWriter;
import ru.yandex.canvas.model.html_builder.InBannerHtmlCreativeWriter;
import ru.yandex.canvas.model.html_builder.LogoGeoPinCreativeHtmlWriter;
import ru.yandex.canvas.model.stillage.StillageInfoConverter;
import ru.yandex.canvas.model.validation.Html5SizeValidator;
import ru.yandex.canvas.model.video.files.MovieAndVideoSourceFactory;
import ru.yandex.canvas.repository.html5.BatchesRepository;
import ru.yandex.canvas.repository.html5.SourcesRepository;
import ru.yandex.canvas.repository.video.AudioFilesRepository;
import ru.yandex.canvas.repository.video.OverlayCreativesRepository;
import ru.yandex.canvas.repository.video.StockVideoAdditionsRepository;
import ru.yandex.canvas.repository.video.VideoAdditionsRepository;
import ru.yandex.canvas.repository.video.VideoConstructorFeedsRepository;
import ru.yandex.canvas.repository.video.VideoConstructorFilesRepository;
import ru.yandex.canvas.repository.video.VideoFilesRepository;
import ru.yandex.canvas.service.AuthRequestParams;
import ru.yandex.canvas.service.AuthRequestParamsImpl;
import ru.yandex.canvas.service.AuthService;
import ru.yandex.canvas.service.AuthServiceHttpProxy;
import ru.yandex.canvas.service.AvatarsService;
import ru.yandex.canvas.service.CreativeBatchDocumentsConsumerFacade;
import ru.yandex.canvas.service.CreativeBatchDocumentsConsumerImpl;
import ru.yandex.canvas.service.CreativeDocumentModifierFacade;
import ru.yandex.canvas.service.CreativesService;
import ru.yandex.canvas.service.DateTimeService;
import ru.yandex.canvas.service.DirectService;
import ru.yandex.canvas.service.DisclaimersService;
import ru.yandex.canvas.service.FileService;
import ru.yandex.canvas.service.GeoPinCreativeBatchDocumentsConsumerImpl;
import ru.yandex.canvas.service.GeoPinCreativeDocumentModifierImpl;
import ru.yandex.canvas.service.HtmlCreativeDocumentModifierImpl;
import ru.yandex.canvas.service.MDSService;
import ru.yandex.canvas.service.MigrationService;
import ru.yandex.canvas.service.MongoMonitorService;
import ru.yandex.canvas.service.PackshotService;
import ru.yandex.canvas.service.PresetsService;
import ru.yandex.canvas.service.PreviewUrlBuilder;
import ru.yandex.canvas.service.RTBHostExportService;
import ru.yandex.canvas.service.SandBoxService;
import ru.yandex.canvas.service.ScreenshooterService;
import ru.yandex.canvas.service.SequenceService;
import ru.yandex.canvas.service.SessionParams;
import ru.yandex.canvas.service.StillageService;
import ru.yandex.canvas.service.StockService;
import ru.yandex.canvas.service.UserAuthService;
import ru.yandex.canvas.service.UserFileService;
import ru.yandex.canvas.service.UserStockService;
import ru.yandex.canvas.service.direct.CreativeDirectUploadHelper;
import ru.yandex.canvas.service.direct.CreativeUploadHelperFacade;
import ru.yandex.canvas.service.direct.GeoPinDirectUploadHelper;
import ru.yandex.canvas.service.direct.Html5CreativeDirectUploadHelper;
import ru.yandex.canvas.service.direct.InBannerDirectUploadHelper;
import ru.yandex.canvas.service.direct.VideoAdditionDirectUploadHelper;
import ru.yandex.canvas.service.html5.Html5BatchesService;
import ru.yandex.canvas.service.html5.Html5CreativesForRtbHostService;
import ru.yandex.canvas.service.html5.Html5CreativesService;
import ru.yandex.canvas.service.html5.Html5SourcesService;
import ru.yandex.canvas.service.html5.PhantomJsCreativesValidator;
import ru.yandex.canvas.service.idea.IdeasService;
import ru.yandex.canvas.service.multitype.OnCreativeServiceProvider;
import ru.yandex.canvas.service.multitype.operation.RebuildOperation;
import ru.yandex.canvas.service.multitype.operation.ScreenshotOperation;
import ru.yandex.canvas.service.multitype.operation.SendToDirectOperation;
import ru.yandex.canvas.service.multitype.operation.SendToRtbHostOperation;
import ru.yandex.canvas.service.preview.CreativePreviewBuilder;
import ru.yandex.canvas.service.preview.CreativePreviewBuilderImpl;
import ru.yandex.canvas.service.rtbhost.helpers.CreativesDspUploadFacade;
import ru.yandex.canvas.service.rtbhost.helpers.VideoMotionService;
import ru.yandex.canvas.service.rtbhost.helpers.creative.CreativesDspUploadHelper;
import ru.yandex.canvas.service.rtbhost.helpers.creative.GeoPinDspUploadHelper;
import ru.yandex.canvas.service.rtbhost.helpers.creative.Html5DspUploadHelper;
import ru.yandex.canvas.service.rtbhost.helpers.creative.InBannerDspUploadHelper;
import ru.yandex.canvas.service.scraper.ScraperService;
import ru.yandex.canvas.service.screenshooters.CreativesScreenshooterHelperService;
import ru.yandex.canvas.service.screenshooters.Html5ScreenshooterHelperService;
import ru.yandex.canvas.service.screenshooters.Html5SourceScreenshooterHelperService;
import ru.yandex.canvas.service.screenshooters.VideoAdditionScreenshooterHelperService;
import ru.yandex.canvas.service.video.AudioService;
import ru.yandex.canvas.service.video.AudioUploadService;
import ru.yandex.canvas.service.video.CmsConversionStatusUpdateService;
import ru.yandex.canvas.service.video.CreativesGenerationService;
import ru.yandex.canvas.service.video.InBannerVideoFilesService;
import ru.yandex.canvas.service.video.MovieService;
import ru.yandex.canvas.service.video.MovieServiceInterface;
import ru.yandex.canvas.service.video.PackshotServiceInterface;
import ru.yandex.canvas.service.video.PackshotUploadingService;
import ru.yandex.canvas.service.video.StaticPreviewService;
import ru.yandex.canvas.service.video.StockVideoAdditionService;
import ru.yandex.canvas.service.video.VhService;
import ru.yandex.canvas.service.video.VideoAdditionValidationService;
import ru.yandex.canvas.service.video.VideoAdditionsService;
import ru.yandex.canvas.service.video.VideoConstructorFeedsService;
import ru.yandex.canvas.service.video.VideoConstructorFilesService;
import ru.yandex.canvas.service.video.VideoCreativesService;
import ru.yandex.canvas.service.video.VideoFileUploadService;
import ru.yandex.canvas.service.video.VideoFileUploadServiceInterface;
import ru.yandex.canvas.service.video.VideoFuzzySearchService;
import ru.yandex.canvas.service.video.VideoGeometryService;
import ru.yandex.canvas.service.video.VideoLimitsService;
import ru.yandex.canvas.service.video.VideoPresetsService;
import ru.yandex.canvas.service.video.VideoPreviewUrlBuilder;
import ru.yandex.canvas.service.video.VideoSoundTrackService;
import ru.yandex.canvas.service.video.files.StockMoviesService;
import ru.yandex.canvas.service.video.files.VideoCategoriesService;
import ru.yandex.canvas.service.video.overlay.OverlayService;
import ru.yandex.direct.asynchttp.FetcherSettings;
import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.blackbox.client.BlackboxClient;
import ru.yandex.direct.bs.dspcreative.configuration.BsDspCreativeConfiguration;
import ru.yandex.direct.bs.dspcreative.service.DspCreativeYtExporter;
import ru.yandex.direct.common.configuration.CommonConfiguration;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.config.DirectConfigFactory;
import ru.yandex.direct.config.DirectConfigPropertySource;
import ru.yandex.direct.env.Environment;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.screenshooter.client.configuration.ScreenShooterConfiguration;
import ru.yandex.direct.screenshooter.client.service.ScreenShooterClient;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmIntegrationImpl;
import ru.yandex.direct.tvm.TvmIntegrationStub;
import ru.yandex.direct.tvm.TvmService;
import ru.yandex.direct.web.auth.blackbox.BlackboxCookieAuthProvider;
import ru.yandex.direct.web.auth.blackbox.BlackboxOauthAuthProvider;
import ru.yandex.inside.passport.blackbox2.Blackbox2;
import ru.yandex.inside.passport.blackbox2.BlackboxRawRequestExecutor;
import ru.yandex.inside.passport.blackbox2.BlackboxRequestExecutorWithRetries;
import ru.yandex.misc.io.http.Timeout;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;

/**
 * @author skirsanov
 */
@Configuration
@Import({
        CommonConfiguration.class,
        BsDspCreativeConfiguration.class,
        ScreenShooterConfiguration.class
})
public class ServiceConfig {

    private static final int MAX_CONNECTIONS = 100;

    @Bean
    static EnvironmentType environmentType() {
        return Environment.getCached();
    }

    @Bean
    static PropertySourcesPlaceholderConfigurer properties(ConfigurableEnvironment env, DirectConfig config) {
        PropertySource<DirectConfig> directConfigSource = new DirectConfigPropertySource("conf", config);

        MutablePropertySources propertySources = env.getPropertySources();
        propertySources.addLast(directConfigSource);

        PropertySourcesPlaceholderConfigurer pspc = new PropertySourcesPlaceholderConfigurer();
        pspc.setIgnoreUnresolvablePlaceholders(true); // иначе падают тесты
        pspc.setPropertySources(propertySources);

        return pspc;
    }

    @Bean
    public PythonRedirect pythonRedirect(@Value("${VIDEO_ADDITIONS_API_URL}") String pythonUrl,
                                         RestTemplate restTemplate)
            throws MalformedURLException {
        return new PythonRedirect(pythonUrl, restTemplate);
    }

    @Bean
    public VideoFuzzySearchService videoFuzzySearchService(@Value("${VIDEO_ADDITIONS_API_URL}") String pythonUrl,
                                                           RestTemplate restTemplate) {
        return new VideoFuzzySearchService(restTemplate, pythonUrl);
    }

    @Bean
    public SequenceService sequenceService(MongoOperations mongoOperation) {
        return new SequenceService(mongoOperation);
    }

    @Bean
    public StillageService stillageService(@Value("${STILLAGE_URL}") String stillageUrl, RestTemplate restTemplate) {
        return new StillageService(stillageUrl, restTemplate);
    }

    @Bean
    public PhantomJsCreativesValidator phantomJsCreativesValidator(@Value("${PHANTOMJS_URL}") String phantomJsUrl,
                                                                   RestTemplate restTemplate) {
        return new PhantomJsCreativesValidator(phantomJsUrl, restTemplate);
    }

    @Bean
    public ScraperService scraperService(@Value("${SCRAPER_URL}") String scraperUrl) {
        return new ScraperService(scraperUrl);
    }

    @Bean
    public InBannerHtmlCreativeWriter inBannerHtmlCreativeWriter(InBannerVideoFilesService inBannerVideoFilesService) {
        return new InBannerHtmlCreativeWriter(inBannerVideoFilesService);
    }

    @Bean
    public CreativesDspUploadHelper creativesDspUploadHelper(CreativeHtmlRtbHostWriter creativeHtmlRtbHostWriter) {
        return new CreativesDspUploadHelper(creativeHtmlRtbHostWriter);
    }

    @Bean
    public GeoPinDspUploadHelper geoPinDspUploadHelper() {
        return new GeoPinDspUploadHelper();
    }

    @Bean
    public InBannerDspUploadHelper inBannerDspUploadHelper(InBannerHtmlCreativeWriter inBannerHtmlCreativeWriter,
                                                           InBannerVideoFilesService inBannerVideoFilesService,
                                                           PresetsService presetsService) {
        return new InBannerDspUploadHelper(inBannerHtmlCreativeWriter, inBannerVideoFilesService, presetsService);
    }

    @Bean
    public CreativesDspUploadFacade creativesDspUploadFacade(CreativesDspUploadHelper creativesDspUploadHelper,
                                                             GeoPinDspUploadHelper geoPinDspUploadHelper,
                                                             InBannerDspUploadHelper inBannerDspUploadHelper) {
        return new CreativesDspUploadFacade(creativesDspUploadHelper, geoPinDspUploadHelper, inBannerDspUploadHelper);
    }

    @Bean
    public GeoPinCreativeHtmlWriter geoPinCreativeHtmlWriter() {
        return new GeoPinCreativeHtmlWriter();
    }

    @Bean
    public ImageGeoPinCreativeHtmlWriter imageGeoPinCreativeHtmlWriter() {
        return new ImageGeoPinCreativeHtmlWriter();
    }

    @Bean
    public LogoGeoPinCreativeHtmlWriter logoGeoPinCreativeHtmlWriter() {
        return new LogoGeoPinCreativeHtmlWriter();
    }

    @Bean
    public PreviewUrlBuilder previewUrlBuilder(DirectService directService,
                                               @Value("${CANVAS_PREVIEW_HOST}") String canvasPreviewHost,
                                               @Value("${WEB_URL}") String webUrl) {
        return new PreviewUrlBuilder(directService, canvasPreviewHost, webUrl);
    }

    @Bean
    public CreativeDirectUploadHelper creativeDirectUploadHelper(PreviewUrlBuilder previewUrlBuilder) {
        return new CreativeDirectUploadHelper(previewUrlBuilder);
    }

    @Bean
    public GeoPinDirectUploadHelper geoPinDirectUploadHelper(PreviewUrlBuilder previewUrlBuilder) {
        return new GeoPinDirectUploadHelper(previewUrlBuilder);
    }

    @Bean
    public InBannerDirectUploadHelper inBannerDirectUploadHelper(PreviewUrlBuilder previewUrlBuilder,
                                                                 InBannerVideoFilesService inBannerVideoFilesService) {
        return new InBannerDirectUploadHelper(previewUrlBuilder, inBannerVideoFilesService);
    }

    @Bean
    public CreativeUploadHelperFacade creativeUploadHelperFacade(CreativeDirectUploadHelper creativeDirectUploadHelper,
                                                                 GeoPinDirectUploadHelper geoPinDirectUploadHelper,
                                                                 InBannerDirectUploadHelper inBannerDirectUploadHelper) {
        return new CreativeUploadHelperFacade(creativeDirectUploadHelper, geoPinDirectUploadHelper,
                inBannerDirectUploadHelper);
    }

    @Bean
    public CreativePreviewBuilder creativePreviewBuilder(GeoPinCreativeHtmlWriter geoPinCreativeHtmlWriter,
                                                         CreativeHtmlPreviewWriter creativeHtmlPreviewWriter,
                                                         InBannerHtmlCreativeWriter inBannerHtmlCreativeWriter) {
        return new CreativePreviewBuilderImpl(geoPinCreativeHtmlWriter, creativeHtmlPreviewWriter,
                inBannerHtmlCreativeWriter);
    }

    @Bean
    public GeoPinCreativeDocumentModifierImpl geoPinCreativeDocumentModifier(
            CreativePreviewBuilder creativePreviewBuilder,
            ScreenshooterService screenshooterService,
            StillageService stillageService,
            ImageGeoPinCreativeHtmlWriter imageGeoPinCreativeHtmlWriter,
            LogoGeoPinCreativeHtmlWriter logoGeoPinCreativeHtmlWriter,
            CreativesScreenshooterHelperService screenshooterHelperService,
            PreviewUrlBuilder previewUrlBuilder,
            PresetsService presetsService) {
        return new GeoPinCreativeDocumentModifierImpl(creativePreviewBuilder, screenshooterService, stillageService,
                imageGeoPinCreativeHtmlWriter, logoGeoPinCreativeHtmlWriter, screenshooterHelperService,
                previewUrlBuilder, presetsService);
    }

    @Bean
    public HtmlCreativeDocumentModifierImpl creativeDocumentModifier(
            CreativePreviewBuilder creativePreviewBuilder,
            ScreenshooterService screenshooterService,
            StillageService stillageService,
            CreativesScreenshooterHelperService creativesScreenshooterHelperService,
            PreviewUrlBuilder previewUrlBuilder,
            PresetsService presetsService) {
        return new HtmlCreativeDocumentModifierImpl(creativePreviewBuilder, screenshooterService, stillageService,
                creativesScreenshooterHelperService, previewUrlBuilder, presetsService);
    }

    @Bean
    public CreativeDocumentModifierFacade creativeDocumentModifierFacade(GeoPinCreativeDocumentModifierImpl geoPinDocumentModifier,
                                                                         HtmlCreativeDocumentModifierImpl creativeDocModifier) {
        return new CreativeDocumentModifierFacade(creativeDocModifier, geoPinDocumentModifier);
    }

    @Bean
    public CreativeBatchDocumentsConsumerImpl creativeBatchDocumentsConsumer(PreviewUrlBuilder previewUrlBuilder) {
        return new CreativeBatchDocumentsConsumerImpl(previewUrlBuilder);
    }

    @Bean
    public GeoPinCreativeBatchDocumentsConsumerImpl geoPinCreativeBatchDocumentsConsumer(PreviewUrlBuilder previewUrlBuilder) {
        return new GeoPinCreativeBatchDocumentsConsumerImpl(previewUrlBuilder);
    }

    @Bean
    public CreativeBatchDocumentsConsumerFacade creativeBatchDocumentsConsumerFacade(CreativeBatchDocumentsConsumerImpl creativeBatchDocumentsConsumer,
                                                                                     GeoPinCreativeBatchDocumentsConsumerImpl geoPinCreativeBatchDocumentsConsumer) {
        return new CreativeBatchDocumentsConsumerFacade(creativeBatchDocumentsConsumer,
                geoPinCreativeBatchDocumentsConsumer);
    }

    @Bean
    public CreativesService creativesService(MongoOperations mongoOperations,
                                             SequenceService sequenceService,
                                             FileService fileService,
                                             ScreenshooterService screenshooterService,
                                             AuthService authService,
                                             RTBHostExportService rtbHostExportService,
                                             CreativesDspUploadFacade creativesDspUploadFacade,
                                             CreativeDocumentModifierFacade creativeDocsFacade,
                                             CreativeBatchDocumentsConsumerFacade consumerFacade) {

        return new CreativesService(mongoOperations, sequenceService, fileService, screenshooterService,
                rtbHostExportService, authService, creativesDspUploadFacade, creativeDocsFacade,
                consumerFacade);
    }

    @Bean
    public CreativeHtmlRtbHostWriter creativeHtmlRtbHostWriter() {
        return new CreativeHtmlRtbHostWriter();
    }

    @Bean
    public CreativeHtmlPreviewWriter creativeHtmlPreviewWriter() {
        return new CreativeHtmlPreviewWriter();
    }

    @Bean
    public DirectService directService(@Value("${DIRECT_URL}") String directUrl,
                                       @Value("${DIRECT_AUTH_TOKEN}") String authToken,
                                       TvmIntegration tvmIntegration,
                                       @Value("${direct_intapi.tvm_app_id}") int directIntApiTvmAppId,
                                       AsyncHttpClient asyncHttpClient) {
        return new DirectService(directUrl, authToken, tvmIntegration, TvmService.fromIdStrict(directIntApiTvmAppId),
                asyncHttpClient);
    }

    @Bean
    public BlackboxCookieAuthProvider blackboxCookieAuthProvider(BlackboxClient blackboxClient) {
        return new BlackboxCookieAuthProvider(blackboxClient);
    }

    @Bean
    public BlackboxOauthAuthProvider blackboxOauthAuthProvider(BlackboxClient blackboxClient) {
        return new BlackboxOauthAuthProvider(blackboxClient);
    }

    @Bean
    @Lazy
    BlackboxClient blackboxClient(DirectConfig directConfig) {
        DirectConfig cfg = directConfig.getBranch("blackbox");

        String blackboxUrl = cfg.getString("endpoint");
        Duration blackboxTimeout = cfg.findDuration("timeout")
                .orElse(Duration.ofSeconds(2));
        int blackboxRetries = cfg.findInt("retries")
                .orElse(3);

        final Timeout timeout = Timeout.milliseconds(blackboxTimeout.toMillis());
        final HttpClient httpClient = ApacheHttpClientUtils.multiThreadedClient(timeout, MAX_CONNECTIONS);
        BlackboxRawRequestExecutor rawExecuter = new BlackboxRawRequestExecutor(blackboxUrl, httpClient);
        BlackboxRequestExecutorWithRetries executorWithRetries =
                new BlackboxRequestExecutorWithRetries(rawExecuter, blackboxRetries);

        Blackbox2 blackbox2 = new Blackbox2(executorWithRetries);
        return new BlackboxClient(blackbox2);
    }

    @Bean
    @Primary
    @RequestScope
    public SessionParams sessionParams(HttpServletRequest httpServletRequest) {
        return new SessionParams(httpServletRequest);
    }

    @Bean
    @RequestScope
    public AuthRequestParams authRequestParams(HttpServletRequest httpServletRequest, SessionParams sessionParams) {
        return new AuthRequestParamsImpl(httpServletRequest, sessionParams);
    }

    @Bean
    public CanvasAuthInterceptor canvasAuthInterceptor(
            @Value("${DEFAULT_AUTH_MODE}") String defaultAuthMode,
            DirectService directService,
            BlackboxCookieAuthProvider blackboxCookieAuthProvider, AuthRequestParams authRequestParams,
            BlackboxOauthAuthProvider blackboxOauthAuthProvider,
            TvmIntegration tvmIntegration,
            @Value("${SANDBOX_HOOK_SECRET}") String sandboxHookSecret,
            @Value("${INTERNAL_API_TOKEN}") String internalApiToken,
            @Value("${spring.profiles.active:Unknown}") String activeProfile,
            EnvironmentType environmentType) {
        CanvasAuthInterceptor.Builder builder = CanvasAuthInterceptor.builder();

        List<AuthorizeBy.AuthType> defaultAuthTypes = Stream.of(defaultAuthMode.split("\\s*,\\s*")).map(
                AuthorizeBy.AuthType::valueOf).collect(
                Collectors.toList());

        CanvasAuthorizer queryStringAuthorizer = new QueryStringAuthorizer(authRequestParams);
        CanvasAuthorizer tvmAuthorizer;

        if ("pythontest".equals(activeProfile)) {
            tvmAuthorizer = queryStringAuthorizer;
        } else {
            tvmAuthorizer = new TvmAuthorizer(tvmIntegration, environmentType, queryStringAuthorizer);
        }

        TvmService tvmService = environmentType.isProductionOrPrestable()
                ? TvmService.BLACKBOX_PROD
                : TvmService.BLACKBOX_MIMINO;

        return builder
                .defaultAuth(defaultAuthTypes)
                .register(AuthorizeBy.AuthType.BLACKBOX,
                        new BlackBoxAuthorizer(blackboxCookieAuthProvider, authRequestParams, tvmIntegration,
                                tvmService))
                .register(AuthorizeBy.AuthType.OAUTH,
                        new OAuthAuthorizer(blackboxOauthAuthProvider, authRequestParams, tvmIntegration, tvmService))
                .register(AuthorizeBy.AuthType.DIRECT_TOKEN, new DirectTokenAuthorizer(directService))
                .register(AuthorizeBy.AuthType.TRUSTED_QUERY_STRING, queryStringAuthorizer)
                .register(AuthorizeBy.AuthType.TVM_TOKEN, tvmAuthorizer)
                .register(AuthorizeBy.AuthType.SANDBOX_SECRET, new SandboxCallbackAuthorizer(sandboxHookSecret))
                .register(AuthorizeBy.AuthType.SANDBOX_CLEANUP_TOKEN,
                        new SandboxCleanupHandleAuthorizer(internalApiToken))
                .build();
    }

    @Bean
    public AuthService authService(@Value("${AUTH_SERVICE_URL}") String authServiceUrl,
                                   AuthRequestParams authRequestParams, TvmIntegration tvmIntegration,
                                   @Value("${direct_intapi.tvm_app_id}") int directIntApiTvmAppId) {
        DefaultAsyncHttpClientConfig config = new DefaultAsyncHttpClientConfig
                .Builder()
                .setConnectTimeout(5 * 1000)
                .setReadTimeout(10 * 1000)
                .setMaxConnections(100)
                .build();
        FetcherSettings fetcherSettings = new FetcherSettings()
                .withParallel(30)
                .withRequestRetries(4)
                .withSoftTimeout(Duration.ofSeconds(3))
                .withRequestTimeout(Duration.ofSeconds(10))
                .withGlobalTimeout(Duration.ofSeconds(20));
        ParallelFetcherFactory fetcherFactory = new ParallelFetcherFactory(new DefaultAsyncHttpClient(config),
                fetcherSettings);
        return new AuthServiceHttpProxy(authRequestParams, authServiceUrl,
                tvmIntegration, TvmService.fromIdStrict(directIntApiTvmAppId),
                fetcherFactory);
    }

    @Bean
    public Html5SizeValidator html5SizeValidator(SessionParams sessionParams, PpcPropertiesSupport ppcPropertiesSupport) {
        return new Html5SizeValidator(sessionParams, ppcPropertiesSupport);
    }

    @Bean
    public VhService vhClient(@Value("${cms_api.url}") String vhBaseUrl,
                              TvmIntegration tvmIntegration,
                              @Value("${cms_api.tvm_app_id}") int cmsTvmAppId) {
        DefaultAsyncHttpClientConfig config = new DefaultAsyncHttpClientConfig
                .Builder()
                .setConnectTimeout(5 * 1000)
                .setReadTimeout(10 * 1000)
                .setMaxConnections(100)
                .build();

        FetcherSettings fetcherSettings = new FetcherSettings()
                .withParallel(20)
                .withRequestRetries(3)
                .withSoftTimeout(Duration.ofSeconds(3))
                .withRequestTimeout(Duration.ofSeconds(10))
                .withGlobalTimeout(Duration.ofSeconds(17));

        ParallelFetcherFactory fetcherFactory = new ParallelFetcherFactory(new DefaultAsyncHttpClient(config),
                fetcherSettings);
        return new VhService(vhBaseUrl, tvmIntegration, TvmService.fromIdStrict(cmsTvmAppId), fetcherFactory);
    }

    @Bean
    public MDSService mdsService(@Value("${MDS_WRITE_URL}") String mdsWriteUrl,
                                 @Value("${MDS_READ_URL}") String mdsReadUrl,
                                 @Value("${MDS_NAMESPACE}") String namespace, //TODO canvas-html5 for production
                                 @Value("${MDS_AUTH_TOKEN}") String mdsToken,
                                 AsyncHttpClient asyncHttpClient) {
        return new MDSService(mdsWriteUrl, mdsReadUrl, namespace, mdsToken, asyncHttpClient);
    }

    @Bean
    public SandBoxService sandBoxService(@Value("${SANDBOX_OAUTH_TOKEN}") String sandboxOAuthToken,
                                         @Value("${SANDBOX_BASE_URL}") String sandboxBaseUrl,
                                         AsyncHttpClient asyncHttpClient) {
        return new SandBoxService(sandboxOAuthToken, sandboxBaseUrl, asyncHttpClient);
    }

    @Bean
    public UserAuthService userAuthService(AuthRequestParams authRequestParams) {
        return new UserAuthService(authRequestParams);
    }

    @Bean
    public PresetsService presetsService(ObjectMapper objectMapper, AuthService authService,
                                         DirectService directService) {
        return new PresetsService(objectMapper, directService);
    }

    @Bean
    public FileService fileService(
            MongoOperations mongoOperations,
            StockService stockService,
            StillageService stillageService,
            AvatarsService avatarsService,
            AuthService authService,
            RestTemplate restTemplate) {
        return new FileService(mongoOperations, stockService, stillageService, authService, avatarsService,
                restTemplate);
    }

    @Bean
    public UserFileService userFileService(MongoOperations mongoOperations,
                                           UserStockService userStockService,
                                           StillageService stillageService,
                                           AvatarsService avatarsService,
                                           UserAuthService userAuthService, SessionParams sessionParams) {
        return new UserFileService(mongoOperations, userStockService, stillageService, userAuthService,
                avatarsService, sessionParams);
    }

    @Bean
    public DisclaimersService disclaimersService() {
        return new DisclaimersService();
    }

    @Bean
    public ScreenshooterService screenshooterService(ScreenShooterClient screenShooterClient,
                                                     AvatarsService avatarsService) {
        return new ScreenshooterService(screenShooterClient, avatarsService);
    }

    @Bean
    public AvatarsService avatarsService(@Value("${AVATARS_WRITE_URL}") String avatarsWriteUrl,
                                         @Value("${AVATARS_READ_URL}") String avatarsReadUrl,
                                         @Value("${AVATARS_NAMESPACE}") String avatarsNamespace,
                                         TvmIntegration tvmIntegration,
                                         @Value("${avatars.tvm_app_id}") int avatarsTvmAppId,
                                         AsyncHttpClient asyncHttpClient) {
        return new AvatarsService(avatarsWriteUrl, avatarsReadUrl, avatarsNamespace, tvmIntegration,
                TvmService.fromIdStrict(avatarsTvmAppId), asyncHttpClient);
    }

    @Bean
    public RTBHostExportService rtbHostExportService(
            DspCreativeYtExporter dspCreativeYtExporter,
            @Value("${RTBHOST_EXPORT_URL}") String rtbHostExportURL,
            @Value("${DSP_CREATIVE_EXPORT_ENABLED:true}") Boolean dspCreativeExportEnabled,
            ObjectMapper objectMapper,
            AsyncHttpClient asyncHttpClient) {
        return new RTBHostExportService(dspCreativeYtExporter, rtbHostExportURL, dspCreativeExportEnabled, objectMapper,
                asyncHttpClient);
    }

    @Bean
    public MongoMonitorService mongoMonitorService(@Value("${MONITORING_TOKEN}") String authToken,
                                                   MongoOperations mongoOperations) {
        return new MongoMonitorService(mongoOperations, authToken);
    }

    @Bean
    public StockService stockService(ObjectMapper objectMapper, AuthService authService) {
        return new StockService(objectMapper, authService);
    }

    @Bean
    public UserStockService userStockService(ObjectMapper objectMapper, UserAuthService authService) {
        return new UserStockService(objectMapper, authService);
    }

    @Bean
    public MigrationService migrationService(PresetsService presetsService,
                                             FileService fileService,
                                             ObjectMapper objectMapper) {
        return new MigrationService(presetsService, fileService, objectMapper);
    }

    @Bean
    public IdeasService ideasService(PresetsService presetsService,
                                     FileService fileService,
                                     MongoOperations mongoOperations,
                                     AuthService authService,
                                     ObjectMapper objectMapper) {
        return new IdeasService(presetsService, fileService, mongoOperations, authService, objectMapper);
    }

    @Bean
    public ObjectMapper objectMapper() {
        ObjectMapper objectMapper = new ObjectMapper();
        objectMapper.disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);
        objectMapper.disable(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS);
        objectMapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
        objectMapper.registerModule(new Jdk8Module());
        return objectMapper;
    }

    @Bean
    @Primary
    public Executor restExecutor() {
        return new ThreadPoolExecutor(20, 200,
                210L, TimeUnit.SECONDS,
                new SynchronousQueue<>(), new ThreadFactoryBuilder()
                .setNameFormat("rest-client-%d").build());

    }

    @Bean
    public Html5SourcesService html5SourcesService(StillageService stillageService, MDSService mdsService,
                                                   SourcesRepository sourcesRepository,
                                                   ParallelFetcherFactory parallelFetcherFactory,
                                                   DirectService directService,
                                                   MovieServiceInterface movieService,
                                                   PhantomJsCreativesValidator phantomJsCreativesValidator,
                                                   Html5SourceScreenshooterHelperService html5SourceScreenshooterHelperService, Html5SizeValidator html5SizeValidator) {
        return new Html5SourcesService(stillageService, mdsService, sourcesRepository,
                parallelFetcherFactory, directService, movieService, phantomJsCreativesValidator,
                html5SourceScreenshooterHelperService, html5SizeValidator);
    }

    @Bean
    public Html5BatchesService html5BatchesService(SequenceService sequenceService,
                                                   BatchesRepository batchesRepository,
                                                   PreviewUrlBuilder previewUrlBuilder,
                                                   Html5SizeValidator html5SizeValidator,
                                                   DirectService directService) {
        return new Html5BatchesService(sequenceService, batchesRepository, previewUrlBuilder, html5SizeValidator,
                directService);
    }

    @Bean
    public SourcesRepository sourcesRepository(MongoOperations mongoOperations) {
        return new SourcesRepository(mongoOperations);
    }

    @Bean
    public BatchesRepository batchesRepository(MongoOperations mongoOperations) {
        return new BatchesRepository(mongoOperations);
    }

    @Bean
    public VideoAdditionsService videoAdditionsService(SequenceService sequenceService,
                                                       VideoAdditionsRepository videoAdditionsRepository,
                                                       VideoPresetsService videoPresetsService,
                                                       @Value("${CANVAS_PREVIEW_HOST}") String canvasPreviewHost,
                                                       MovieServiceInterface movieService,
                                                       PackshotServiceInterface packshotService,
                                                       DirectService directService, DateTimeService dateTimeService,
                                                       AudioService audioService,
                                                       VideoAdditionScreenshooterHelperService videoAdditionScreenshooterHelperService, VideoPreviewUrlBuilder videoPreviewUrlBuilder) {
        return new VideoAdditionsService(sequenceService, videoAdditionsRepository, videoPresetsService,
                canvasPreviewHost, movieService, packshotService, directService, dateTimeService,
                audioService, videoAdditionScreenshooterHelperService, videoPreviewUrlBuilder);
    }

    @Bean
    public VideoPreviewUrlBuilder videoPreviewUrlBuilder(DirectService directService,
                                                         @Value("${CANVAS_PREVIEW_HOST}") String canvasPreviewHost,
                                                         @Value("${WEB_URL}") String webUrl) {
        return new VideoPreviewUrlBuilder(directService, canvasPreviewHost, webUrl);
    }

    @Bean
    public StockVideoAdditionService stockVideoAdditionService(
            StockVideoAdditionsRepository stockVideoAdditionsRepository) {
        return new StockVideoAdditionService(stockVideoAdditionsRepository);
    }

    @Bean
    public VideoAdditionValidationService videoAdditionValidationService(VideoPresetsService videoPresetsService,
                                                                         MovieServiceInterface movieService,
                                                                         PackshotServiceInterface packshotService,
                                                                         AudioService audioService,
                                                                         DirectService directService) {
        return new VideoAdditionValidationService(videoPresetsService, movieService, packshotService, audioService,
                directService);
    }

    @Bean
    public VideoAdditionsRepository videoAdditionsRepository(MongoOperations mongoOperations) {
        return new VideoAdditionsRepository(mongoOperations);
    }

    @Bean
    public ParallelFetcherFactory parallelFetcherFactory() {
        DefaultAsyncHttpClientConfig config = new DefaultAsyncHttpClientConfig
                .Builder()
                .setConnectTimeout(30 * 1000)
                .setReadTimeout(5 * 1000)
                .setMaxConnections(100)
                .build();

        return new ParallelFetcherFactory(new DefaultAsyncHttpClient(config), new FetcherSettings());
    }

    @Bean
    public Html5DspUploadHelper html5DspUploadHelper(DirectService directService) {
        return new Html5DspUploadHelper(directService);
    }

    @Bean
    public Html5CreativeDirectUploadHelper html5CreativeDirectUploadHelper(PreviewUrlBuilder previewUrlBuilder) {
        return new Html5CreativeDirectUploadHelper(previewUrlBuilder);
    }

    @Bean
    public Html5CreativesService html5CreativesService(BatchesRepository batchesRepository,
                                                       Html5SourcesService sourcesService,
                                                       DirectService directService,
                                                       RTBHostExportService rtbHostExportService,
                                                       Html5DspUploadHelper html5DspUploadHelper,
                                                       Html5CreativeDirectUploadHelper html5CreativeUploadData) {
        return new Html5CreativesService(batchesRepository, sourcesService, directService, rtbHostExportService,
                html5DspUploadHelper, html5CreativeUploadData);
    }

    @Bean
    public Html5CreativesForRtbHostService html5CreativesForRtbHostService(
            BatchesRepository batchesRepository,
            Html5SourcesService sourcesService,
            DirectService directService,
            RTBHostExportService rtbHostExportService,
            Html5DspUploadHelper html5DspUploadHelper,
            Html5CreativeDirectUploadHelper html5CreativeUploadData) {
        return new Html5CreativesForRtbHostService(batchesRepository, sourcesService, directService,
                rtbHostExportService, html5DspUploadHelper, html5CreativeUploadData);
    }

    @Bean
    public VideoPresetsService videoPresetsService(VideoLimitsService videoLimitsService, DirectService directService,
                                                   AuthRequestParams authRequestParams) {
        return new VideoPresetsService(videoLimitsService, directService, authRequestParams);
    }

    @Bean
    public StockMoviesService mergedFilesStockService(VideoFuzzySearchService videoFuzzySearchService,
                                                      MovieAndVideoSourceFactory movieAndVideoSourceFactory) {
        return new StockMoviesService(videoFuzzySearchService, movieAndVideoSourceFactory);
    }

    @Bean
    public VideoFilesRepository videoFilesRepository(MongoOperations mongoOperations, DateTimeService dateTimeService) {
        return new VideoFilesRepository(mongoOperations, dateTimeService);
    }

    @Bean
    public AudioFilesRepository audioFilesRepository(MongoOperations mongoOperations, DateTimeService dateTimeService) {
        return new AudioFilesRepository(mongoOperations, dateTimeService);
    }

    @Bean
    public VideoConstructorFilesRepository videoConstructorFilesRepository(MongoOperations mongoOperations,
                                                                           DateTimeService dateTimeService) {
        return new VideoConstructorFilesRepository(mongoOperations, dateTimeService);
    }

    @Bean
    public VideoConstructorFeedsRepository videoConstructorFeedsRepository(MongoOperations mongoOperations) {
        return new VideoConstructorFeedsRepository(mongoOperations);
    }

    @Bean
    public OverlayCreativesRepository overlayCreativesRepository(MongoOperations mongoOperations) {
        return new OverlayCreativesRepository(mongoOperations);
    }

    @Bean
    public DateTimeService dateTimeService() {
        return new DateTimeService();
    }

    @Bean
    public VideoGeometryService videoGeometryService(VideoLimitsService limitsService) {
        return new VideoGeometryService(limitsService);
    }

    @Bean
    public MovieServiceInterface movieService(VideoFilesRepository videoFilesRepository,
                                              StockMoviesService stockMoviesService,
                                              VideoFileUploadServiceInterface fileUploadService,
                                              StillageService stillageService, VideoLimitsService videoLimitsService,
                                              DateTimeService dateTimeService,
                                              DirectService directService, VideoPresetsService videoPresetsService,
                                              CmsConversionStatusUpdateService cmsConversionStatusUpdateService,
                                              MovieAndVideoSourceFactory movieAndVideoSourceFactory,
                                              VideoGeometryService videoGeometryService) {
        return new MovieService(videoFilesRepository, stockMoviesService, fileUploadService,
                stillageService, videoLimitsService, dateTimeService, directService, videoPresetsService,
                cmsConversionStatusUpdateService, movieAndVideoSourceFactory, videoGeometryService);
    }

    @Bean
    public RestTemplate restTemplate() {
        Logger logger = LoggerFactory.getLogger(RestTemplateLoggerInterceptor.LOGGER_NAME);

        RestTemplate restTemplate = new RestTemplate(new BufferingClientHttpRequestFactory(
                new SimpleClientHttpRequestFactory()));

        RestTemplateLoggerInterceptor loggerInterceptor = new RestTemplateLoggerInterceptor();
        if (loggerInterceptor.getLogger().isDebugEnabled()) {
            List<ClientHttpRequestInterceptor> interceptors
                    = restTemplate.getInterceptors();
            if (CollectionUtils.isEmpty(interceptors)) {
                interceptors = new ArrayList<>();
            }
            interceptors.add(loggerInterceptor);
            restTemplate.setInterceptors(interceptors);
        }

        restTemplate.setErrorHandler(new DefaultResponseErrorHandler() {
            @Override
            public void handleError(ClientHttpResponse response) throws IOException {
                logger.warn("Response error status: {} body: {}", response.getRawStatusCode(),
                        FileCopyUtils.copyToString(new InputStreamReader(response.getBody())));
                super.handleError(response);
            }
        });

        // После подключения библиотеки smart-client, вместе с ней притянулся jackson-dataformat-xml
        // Из-за этого по дефолту RestTemplate стал конвертировать объекты в xml, а не в json
        // (MessageConverter для xml стоит раньше, чем для json в списке, а RestTemplate берет первый подходящий,
        // если ему явно не указывать Content-type)
        // Поэтому удаляем конвертер для xml
        // Если в дальнейшем он понадобится, скорее всего можно просто поменять их местами с json-конвертером в списке,
        // но лучше всего всегда явно указывать Content-type при составлении запроса
        restTemplate.getMessageConverters().removeIf(
                m -> m.getClass().getName().equals(MappingJackson2XmlHttpMessageConverter.class.getName()));
        restTemplate.getMessageConverters().add(protobufHttpMessageConverter());

        return restTemplate;
    }

    @Bean
    ProtobufHttpMessageConverter protobufHttpMessageConverter() {
        return new ProtobufHttpMessageConverter();
    }

    @Bean
    public VideoCreativesService videoCreativesService(
            MovieServiceInterface movieService, AudioService audioService, PackshotServiceInterface packshotService,
            VideoAdditionsService videoAdditionsService,
            VideoAdditionsRepository videoAdditionsRepository, DirectService directService,
            RTBHostExportService rtbHostExportService, StockVideoAdditionsRepository stockVideoAdditionsRepository,
            VideoPresetsService videoPresetsService,
            DateTimeService dateTimeService,
            CmsConversionStatusUpdateService cmsConversionStatusUpdateService,
            VideoAdditionDirectUploadHelper videoAdditionDirectUploadHelper) {
        return new VideoCreativesService(movieService, audioService, packshotService, videoAdditionsService, videoAdditionsRepository,
                directService, rtbHostExportService, stockVideoAdditionsRepository, cmsConversionStatusUpdateService,
                videoPresetsService, videoAdditionDirectUploadHelper, dateTimeService);
    }

    @Bean
    public VideoMotionService videoMotionService(RTBHostExportService rtbHostExportService) {
        return new VideoMotionService(rtbHostExportService);
    }

    @Bean
    public CreativesGenerationService creativesGenerationService(StockMoviesService stockMoviesService,
                                                                 VideoPresetsService presetsService,
                                                                 VideoAdditionsService videoAdditionsService,
                                                                 VideoAdditionsRepository videoAdditionsRepository,
                                                                 StockVideoAdditionsRepository stockVideoAdditionsRepository,
                                                                 VideoCreativesService videoCreativesService) {
        return new CreativesGenerationService(stockMoviesService,
                presetsService,
                videoAdditionsService,
                videoAdditionsRepository,
                stockVideoAdditionsRepository,
                videoCreativesService);
    }

    @Bean
    public StockVideoAdditionsRepository stockVideoAddtitionsRepository(MongoOperations mongoOperations) {
        return new StockVideoAdditionsRepository(mongoOperations);
    }

    @Bean
    public PackshotUploadingService packshotUploadingService(
            VideoFilesRepository videoFilesRepository, AvatarsService avatarsService, SessionParams sessionParams,
            VideoPresetsService videoPresetsService, DirectService directService,
            VideoGeometryService videoGeometryService) {
        return new PackshotUploadingService(videoFilesRepository, avatarsService, sessionParams, videoPresetsService,
                directService, videoGeometryService);
    }

    @Bean
    public AudioUploadService audioUploadService(AudioFilesRepository audioFilesRepository,
                                                 VideoLimitsService videoLimitsService,
                                                 DirectService directService) {
        return new AudioUploadService(audioFilesRepository, videoLimitsService, directService);
    }

    @Bean
    public StillageInfoConverter stillageInfoConverter(ObjectMapper objectMapper) {
        return new StillageInfoConverter(objectMapper);
    }

    @Bean
    public MovieAndVideoSourceFactory movieAndVideoSourceFactory(StillageInfoConverter stillageInfoConverter) {
        return new MovieAndVideoSourceFactory(stillageInfoConverter);
    }

    @Bean
    public VideoConstructorFilesService videoConstructorFilesService(
            VideoConstructorFilesRepository videoConstructorFilesRepository,
            VideoLimitsService videoLimitsService,
            DirectService directService,
            VideoConstructorFeedsRepository videoConstructorFeedsRepository,
            VideoConstructorFeedsService videoConstructorFeedsService,
            @Value("${SANDBOX_HOOK_SECRET}") String hookSecret,
            @Value("${CANVAS_VA_API_BASE_URL}") String canvasVideoApiBaseUrl,
            SandBoxService sandBoxService, VideoGeometryService videoGeometryService) {
        return new VideoConstructorFilesService(videoConstructorFilesRepository, sandBoxService, videoLimitsService,
                directService, videoConstructorFeedsRepository, videoConstructorFeedsService, hookSecret,
                canvasVideoApiBaseUrl, videoGeometryService);
    }

    @Bean
    public VideoConstructorFeedsService videoConstructorFeedsService(
            VideoConstructorFeedsRepository videoConstructorFeedsRepository,
            StillageService stillageService,
            AvatarsService avatarsService,
            FileService fileService,
            MovieServiceInterface movieService,
            VideoFileUploadServiceInterface fileUploadService,
            VideoLimitsService videoLimitsService,
            DirectService directService,
            AudioService audioService,
            RestTemplate restTemplate, VideoGeometryService videoGeometryService) {
        return new VideoConstructorFeedsService(videoConstructorFeedsRepository, stillageService,
                avatarsService, fileService, movieService, fileUploadService, videoLimitsService, directService,
                audioService, restTemplate, videoGeometryService);
    }

    @Bean
    public VideoFileUploadServiceInterface videoFileUploadService(
            SandBoxService sandBoxService,
            @Value("${SANDBOX_HOOK_SECRET}") String hookSecret,
            @Value("${CANVAS_VA_API_BASE_URL}") String canvasVaApiBaseUrl,
            VideoFilesRepository videoFilesRepository,
            VideoLimitsService videoLimitsService,
            AvatarsService avatarsService,
            DirectService directService,
            VideoPresetsService videoPresetsService,
            StillageService stillageService,
            StillageInfoConverter stillageInfoConverter,
            VhService vhClient, VideoGeometryService videoGeometryService) {
        return new VideoFileUploadService(sandBoxService, hookSecret, canvasVaApiBaseUrl, videoFilesRepository,
                videoLimitsService, avatarsService, directService, videoPresetsService, stillageService,
                stillageInfoConverter, vhClient, videoGeometryService);
    }

    @Bean
    public AudioService audioService(AudioFilesRepository audioFilesRepository, StillageService stillageService,
                                     AudioUploadService audioUploadService,
                                     @Value("${SANDBOX_HOOK_SECRET}") String hookSecret,
                                     @Value("${CANVAS_VA_API_BASE_URL}") String canvasVaApiBaseUrl,
                                     SandBoxService sandBoxService,
                                     VideoLimitsService videoLimitsService) {
        return new AudioService(audioFilesRepository, stillageService, audioUploadService, sandBoxService,
                videoLimitsService, hookSecret, canvasVaApiBaseUrl);
    }

    @Bean
    public OverlayService overlayService(OverlayCreativesRepository overlayCreativesRepository,
                                         StillageService stillageService,
                                         SequenceService sequenceService,
                                         DateTimeService dateTimeService) {
        return new OverlayService(overlayCreativesRepository, stillageService, sequenceService, dateTimeService);
    }

    @Bean
    public VideoSoundTrackService videoSoundTrackService(StockMoviesService stockMoviesService,
                                                         SessionParams sessionParams,
                                                         VideoFilesRepository videoFilesRepository) {
        return new VideoSoundTrackService(stockMoviesService, sessionParams, videoFilesRepository);
    }

    @Bean
    public PackshotService packshotService(PackshotUploadingService fileUploadService, SessionParams sessionParams,
                                           VideoFilesRepository videoFilesRepository, StillageService stillageService,
                                           VideoLimitsService videoLimitsService, DirectService directService,
                                           VideoPresetsService videoPresetsService,
                                           VideoGeometryService videoGeometryService) {
        return new PackshotService(videoFilesRepository, sessionParams, fileUploadService, stillageService,
                videoLimitsService, directService, videoPresetsService, videoGeometryService);
    }

    @Bean
    public VideoLimitsService videoLimitsService(AuthRequestParams authRequestParams, DirectService directService) {
        return new VideoLimitsService(authRequestParams, directService);
    }

    @Bean
    public VideoCategoriesService videoCategoriesService() throws IOException {
        return new VideoCategoriesService();
    }

    @Bean
    public DirectConfig config() {
        return DirectConfigFactory.getConfig();
    }

    @Bean
    TaskScheduler liveConfigChangeTaskScheduler() {
        return new ThreadPoolTaskScheduler();
    }

    @Bean
    public TvmIntegration tvmIntegration(DirectConfig directConfig,
                                         @Qualifier("liveConfigChangeTaskScheduler") TaskScheduler liveConfigChangeTaskScheduler) {
        boolean enabled = directConfig.findBoolean("tvm.enabled").orElse(true);
        if (enabled) {
            return TvmIntegrationImpl.create(directConfig, liveConfigChangeTaskScheduler);
        } else {
            return new TvmIntegrationStub();
        }
    }

    /**
     * Чтобы одни и теже сервисы реализующие базовые операции над креативами не копипастить в конструкторы операций
     */
    @Bean
    @Lazy
    public OnCreativeServiceProvider onCreativeServiceProvider(Html5CreativesService html5CreativesService,
                                                               CreativesService creativesService,
                                                               VideoAdditionsService videoAdditionsService,
                                                               StockVideoAdditionService stockVideoAdditionService) {
        return new OnCreativeServiceProvider(html5CreativesService, creativesService, videoAdditionsService,
                stockVideoAdditionService);
    }

    @Bean
    @Lazy
    public ScreenshotOperation screenshotOperation(OnCreativeServiceProvider onCreativeServiceProvider,
                                                   Html5BatchesService html5BatchesService,
                                                   Html5ScreenshooterHelperService screenshooterHelperService,
                                                   CreativesScreenshooterHelperService creativesScreenshooterHelperService, VideoAdditionScreenshooterHelperService videoAdditionScreenshooterHelperService) {
        return new ScreenshotOperation(onCreativeServiceProvider,
                screenshooterHelperService, creativesScreenshooterHelperService,
                videoAdditionScreenshooterHelperService);
    }

    @Bean
    @Lazy
    public SendToRtbHostOperation sendToRtbHostOperation(CreativesDspUploadFacade creativesDspUploadFacade,
                                                         CreativesService creativesService,
                                                         VideoAdditionsService videoAdditionsService,
                                                         VideoCreativesService videoCreativesService,
                                                         Html5CreativesForRtbHostService html5CreativesForRtbHostService,
                                                         RTBHostExportService rtbHostExportService,
                                                         Html5SourcesService sourcesService,
                                                         VideoAdditionsRepository videoAdditionsRepository,
                                                         Html5DspUploadHelper html5DspUploadHelper,
                                                         StockVideoAdditionService stockVideoAdditionService,
                                                         StockVideoAdditionsRepository stockVideoAdditionsRepository) {

        OnCreativeServiceProvider onCreativeServiceProvider = new OnCreativeServiceProvider(videoAdditionsService,
                html5CreativesForRtbHostService, creativesService, stockVideoAdditionService);

        return new SendToRtbHostOperation(onCreativeServiceProvider, creativesDspUploadFacade, videoCreativesService,
                rtbHostExportService, sourcesService, html5DspUploadHelper, videoAdditionsRepository,
                stockVideoAdditionsRepository);
    }

    @Bean
    @Lazy
    public SendToDirectOperation sendToDirectOperation(VideoAdditionsService videoAdditionsService,
                                                       Html5CreativeDirectUploadHelper html5CreativeDirectUploadHelper,
                                                       CreativeUploadHelperFacade creativeUploadHelperFacade,
                                                       Html5CreativesService html5CreativesService,
                                                       CreativesService creativesService,
                                                       DirectService directService,
                                                       VideoAdditionDirectUploadHelper videoAdditionDirectUploadHelper) {

        OnCreativeServiceProvider onCreativeServiceProvider = new OnCreativeServiceProvider(videoAdditionsService,
                html5CreativesService, creativesService);

        return new SendToDirectOperation(onCreativeServiceProvider,
                html5CreativeDirectUploadHelper, creativeUploadHelperFacade,
                videoAdditionDirectUploadHelper, directService);
    }

    @Bean
    @Lazy
    public RebuildOperation rebuildOperation(OnCreativeServiceProvider onCreativeServiceProvider,
                                             VideoCreativesService videoCreativesService,
                                             Html5CreativesService html5CreativesService) {
        return new RebuildOperation(onCreativeServiceProvider, videoCreativesService, html5CreativesService);
    }

    @Bean
    public InBannerVideoFilesService inBannerVideoFilesService(VideoFileUploadServiceInterface videoFileUploadService,
                                                               VideoFilesRepository videoFilesRepository,
                                                               StillageService stillageService,
                                                               DateTimeService dateTimeService,
                                                               PresetsService presetsService,
                                                               StockMoviesService stockMoviesService,
                                                               CmsConversionStatusUpdateService cmsConversionStatusUpdateService) {
        return new InBannerVideoFilesService(videoFileUploadService, videoFilesRepository, stillageService,
                dateTimeService, presetsService, stockMoviesService, cmsConversionStatusUpdateService);
    }

    @Bean
    public CmsConversionStatusUpdateService cmsConversionStatusUpdateService(VhService vhClient,
                                                                             VideoFileUploadServiceInterface fileUploadService,
                                                                             VideoFilesRepository videoFilesRepository) {
        return new CmsConversionStatusUpdateService(vhClient, fileUploadService, videoFilesRepository);
    }

    @Bean
    public VideoAdditionScreenshooterHelperService videoAdditionScreenshooterHelperService(VideoPresetsService presetsService,
                                                                                           DirectService directService,
                                                                                           ScreenshooterService screenshooterService,
                                                                                           VideoAdditionsRepository videoAdditionsRepository,
                                                                                           VideoAdditionDirectUploadHelper videoAdditionDirectUploadHelper, StaticPreviewService staticPreviewService) {
        return new VideoAdditionScreenshooterHelperService(presetsService,
                screenshooterService, videoAdditionsRepository, directService, videoAdditionDirectUploadHelper,
                staticPreviewService);
    }

    @Bean
    public Html5ScreenshooterHelperService html5ScreenshooterHelperService(DirectService directService,
                                                                           Html5CreativeDirectUploadHelper html5CreativeDirectUploadHelper,
                                                                           Html5SourceScreenshooterHelperService html5SourceScreenshooterHelperService,
                                                                           BatchesRepository batchesRepository) {
        return new Html5ScreenshooterHelperService(
                html5CreativeDirectUploadHelper, directService, html5SourceScreenshooterHelperService,
                batchesRepository);
    }

    @Bean
    public Html5SourceScreenshooterHelperService html5SourceScreenshooterHelperService(SourcesRepository sourcesRepository,
                                                                                       ScreenshooterService screenShooterService,
                                                                                       DirectService directService) {
        return new Html5SourceScreenshooterHelperService(screenShooterService, sourcesRepository, directService);
    }

    @Bean
    public CreativesScreenshooterHelperService creativesScreenshooterHelperService(ScreenshooterService screenshooterService,
                                                                                   DirectService directService,
                                                                                   CreativeUploadHelperFacade creativeUploadHelperFacade,
                                                                                   MongoOperations mongoOperation,
                                                                                   InBannerHtmlCreativeWriter inBannerHtmlCreativeWriter) {
        return new CreativesScreenshooterHelperService(screenshooterService, mongoOperation, directService,
                creativeUploadHelperFacade, inBannerHtmlCreativeWriter);
    }

    @Bean
    public StaticPreviewService staticPreviewService(MovieServiceInterface movieService,
                                                     VideoPresetsService videoPresetsService) {
        return new StaticPreviewService(movieService, videoPresetsService);
    }

    @Bean
    public VideoAdditionDirectUploadHelper videoAdditionDirectUploadHelper(AudioService audioService,
                                                                           MovieServiceInterface movieService,
                                                                           VideoPreviewUrlBuilder videoPreviewUrlBuilder,
                                                                           DirectService directService) {
        return new VideoAdditionDirectUploadHelper(videoPreviewUrlBuilder, audioService, movieService, directService);
    }

}
