package ru.yandex.canvas.configs;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;

import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.BeanPostProcessor;

import ru.yandex.canvas.service.OnCreativeService;
import ru.yandex.canvas.service.multitype.CreativeType;
import ru.yandex.canvas.service.multitype.MultiCreativeTypeOperation;
import ru.yandex.canvas.service.multitype.SingleCreativeTypeOperation;
import ru.yandex.canvas.service.multitype.SingleCreativeTypeOperationProvider;

/**
 * Построцессор для бинов реализующих интерфейс {@link MultiCreativeTypeOperation} - операции обрабатывающие
 * креативы разных типов по их Id.
 * Ищет методы аннотированные {@link SingleCreativeTypeOperationProvider}, получает
 * из них объекты операций, инициализирует их и инициализирует ими мульти-типовую операцию, проверяя
 * что мульти-типовая операция правильно описана
 */
public class SingleTypeCreativeIdsOperationBeanPostProcessor implements BeanPostProcessor {
    @Override
    public Object postProcessBeforeInitialization(Object bean, String beanName) throws BeansException {
        if (bean instanceof MultiCreativeTypeOperation) {
            initializeBean(bean, beanName);
        }
        return bean;
    }

    private void initializeBean(Object bean, String beanName) {
        Map<CreativeType, SingleCreativeTypeOperation> operationByType = new HashMap<>();
        Map<CreativeType, OnCreativeService> serviceForType = new HashMap<>();
        MultiCreativeTypeOperation multiTypeOperation = ((MultiCreativeTypeOperation) bean);
        for (Method method : bean.getClass().getMethods()) {
            if (method.isAnnotationPresent(SingleCreativeTypeOperationProvider.class)) {
                CreativeType type = method.getAnnotation(SingleCreativeTypeOperationProvider.class).value();
                if (operationByType.containsKey(type)) {
                    throw new IllegalStateException(beanName + ": already has operation for type " + type);
                }
                SingleCreativeTypeOperation operation;
                try {
                    operation = (SingleCreativeTypeOperation) method.invoke(bean);
                } catch (IllegalAccessException | InvocationTargetException e) {
                    throw new RuntimeException(
                            "Can't get operation for type " + type + " in " + beanName + ": " + e.getMessage(), e);
                }
                if (!type.equals(operation.worksOn())) {
                    throw new IllegalStateException(
                            beanName + ": operation works on " + operation.worksOn() + " while annotated " + type);
                }
                OnCreativeService service = ((MultiCreativeTypeOperation) bean).provideService(type); // CCE intended
                operation.init(service, LoggerFactory.getLogger(operation.getClass()));
                serviceForType.put(type, service);
                operationByType.put(type, operation);
            }
        }
        if (operationByType.isEmpty()) {
            throw new IllegalStateException(
                    "Multitype operation : " + beanName + " declared without any supported type");
        }
        multiTypeOperation.init(operationByType, serviceForType);
    }

    @Override
    public Object postProcessAfterInitialization(Object bean, String beanName) throws BeansException {
        return bean;
    }
}
