package ru.yandex.canvas.configs;

import java.util.Arrays;
import java.util.List;
import java.util.Locale;

import javax.servlet.http.HttpServletRequest;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.servlet.i18n.AcceptHeaderLocaleResolver;

/**
 * Определяет локаль запроса.
 * Мы хотим чтобы локаль совпадала с фронтэндом, поэтому ждём что она будет передана в хедере.
 * Если в запросе приходит X-Detected-Locale, то используем её как приоритетную. Значит фронтэнд вычислил и валидировал.
 * Если X-Detected-Locale не пришла, то используем Accept-Language. Такие запросы приходят с node.js
 * Если локаль никто не передал, то ответим дефолтным языком.
 */
public class WebLocaleResolver extends AcceptHeaderLocaleResolver {
    private static final Logger logger = LoggerFactory.getLogger(WebLocaleResolver.class);
    public static final String DETECTED_LOCALE_HEADER_NAME = "X-Detected-Locale";
    public static final Locale DEFAULT_LOCALE = new Locale("en");
    public static final List<Locale> LOCALES = Arrays.asList(new Locale("en"),
            new Locale("ru"), new Locale("tr"), new Locale("uk"));

    @Override
    public Locale resolveLocale(HttpServletRequest request) {
        String detectedLocale = request.getHeader(DETECTED_LOCALE_HEADER_NAME);
        if (detectedLocale != null) {
            Locale locale = new Locale(detectedLocale);
            return new Locale(locale.getLanguage());
        }
        String acceptLanguageHeader = request.getHeader("Accept-Language");
        if (acceptLanguageHeader != null) {
            try {
                //Иногда присылают язык "en_US" вместо "en-us". Мы хотим уметь обрабатывать такие случаи
                var list = Locale.LanguageRange.parse(acceptLanguageHeader.replaceAll("_", "-"));
                Locale locale = Locale.lookup(list, LOCALES);
                return locale == null ? DEFAULT_LOCALE : locale;
            } catch (IllegalArgumentException e) {
                logger.warn("Accept-Language parse problem", e);
                return DEFAULT_LOCALE;
            }
        }
        return DEFAULT_LOCALE;
    }
}
