package ru.yandex.canvas.configs;

import javax.servlet.MultipartConfigElement;

import org.hibernate.validator.messageinterpolation.ResourceBundleMessageInterpolator;
import org.hibernate.validator.resourceloading.PlatformResourceBundleLocator;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.web.servlet.FilterRegistrationBean;
import org.springframework.boot.web.servlet.MultipartConfigFactory;
import org.springframework.boot.web.servlet.ServletRegistrationBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.context.annotation.Primary;
import org.springframework.http.MediaType;
import org.springframework.util.unit.DataSize;
import org.springframework.validation.Validator;
import org.springframework.validation.beanvalidation.LocalValidatorFactoryBean;
import org.springframework.web.servlet.LocaleResolver;
import org.springframework.web.servlet.config.annotation.ContentNegotiationConfigurer;
import org.springframework.web.servlet.config.annotation.InterceptorRegistry;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurer;

import ru.yandex.canvas.model.validation.ValidateOnlyAvailableElementsAndCreatives;
import ru.yandex.direct.common.admin.engine.AdminServlet;
import ru.yandex.direct.common.configuration.CommonConfiguration;
import ru.yandex.direct.common.metrics.MetricsInterceptor;
import ru.yandex.direct.common.tracing.TraceContextFilter;
import ru.yandex.direct.solomon.SolomonMonitoringServlet;

/**
 * @author pupssman
 */
@Configuration
@Import(CommonConfiguration.class)
public class WebMvcConfig implements WebMvcConfigurer {
    @Autowired
    private CanvasTvmInterceptor tvmInterceptor;

    @Autowired
    private CanvasAuthInterceptor authInterceptor;

    @Autowired
    private MetricsInterceptor metricsInterceptor;

    @Bean
    public XCanvasRequestIdHeaderAppenderFilter xCanvasRequestIdHeaderAppenderFilter() {
        return new XCanvasRequestIdHeaderAppenderFilter();
    }

    @Bean
    public SingleTypeCreativeIdsOperationBeanPostProcessor singleTypeCreativeIdsOperationBeanPostProcessor() {
        return new SingleTypeCreativeIdsOperationBeanPostProcessor();
    }

    @Bean
    public ServletRegistrationBean monitoringServletRegistrationBean() {
        return new ServletRegistrationBean(new SolomonMonitoringServlet(), "/monitoring");
    }

    @Bean
    public ServletRegistrationBean adminServletRegistrationBean() {
        return new ServletRegistrationBean(new AdminServlet(), "/admin");
    }


    @Bean
    public LocalValidatorFactoryBean localValidatorFactoryBean() {
        final LocalValidatorFactoryBean validator = new LocalValidatorFactoryBean();

        validator.setMessageInterpolator(new ResourceBundleMessageInterpolator(
                new PlatformResourceBundleLocator("localized.ValidationMessages")));


        validator.setTraversableResolver(
                new ValidateOnlyAvailableElementsAndCreatives());

        return validator;
    }

    @Bean
    public FilterRegistrationBean filterRegistrationBean(TraceContextFilter traceContextFilter) {

        FilterRegistrationBean registrationBean = new FilterRegistrationBean();
        registrationBean.setFilter(traceContextFilter);
        registrationBean.addUrlPatterns("/*");

        return registrationBean;
    }

    @Bean
    public LocaleResolver localeResolver() {
        return new WebLocaleResolver();
    }

    @Override
    public void configureContentNegotiation(ContentNegotiationConfigurer configurer) {
        configurer.favorPathExtension(false)
                .defaultContentType(MediaType.APPLICATION_JSON_UTF8);
    }

    @Bean
    public MultipartConfigElement multipartConfigElement() {

        MultipartConfigFactory factory = new MultipartConfigFactory();
        factory.setMaxFileSize(DataSize.ofMegabytes(250));
        factory.setMaxRequestSize(DataSize.ofMegabytes(250));

        return factory.createMultipartConfig();
    }

    @Override
    @Bean
    @Primary
    public Validator getValidator() {
        return localValidatorFactoryBean();
    }


    @Override
    public void addInterceptors(InterceptorRegistry registry) {
        registry.addInterceptor(tvmInterceptor);
        registry.addInterceptor(authInterceptor);
        registry.addInterceptor(metricsInterceptor);
    }
}
