package ru.yandex.canvas.configs.auth;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;

import org.springframework.http.HttpHeaders;
import org.springframework.security.core.AuthenticationException;

import ru.yandex.canvas.exceptions.AuthException;
import ru.yandex.canvas.exceptions.UnauthorizedException;
import ru.yandex.canvas.service.AuthRequestParams;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;
import ru.yandex.direct.web.auth.blackbox.BlackboxCookieAuth;
import ru.yandex.direct.web.auth.blackbox.BlackboxCookieAuthProvider;
import ru.yandex.direct.web.auth.blackbox.BlackboxCookieAuthRequest;
import ru.yandex.direct.web.auth.blackbox.BlackboxCookieCredentials;

public class BlackBoxAuthorizer implements CanvasAuthorizer {
    private static final String X_REAL_IP_HEADER = "X-Real-IP";
    private final BlackboxCookieAuthProvider blackboxCookieAuthProvider;
    private final AuthRequestParams authRequestParams;
    private final TvmIntegration tvmIntegration;
    private final TvmService tvmService;

    public BlackBoxAuthorizer(BlackboxCookieAuthProvider blackboxCookieAuthProvider,
                              AuthRequestParams authRequestParams,
                              TvmIntegration tvmIntegration,
                              TvmService tvmService)
    {
        this.blackboxCookieAuthProvider = blackboxCookieAuthProvider;
        this.authRequestParams = authRequestParams;
        this.tvmIntegration = tvmIntegration;
        this.tvmService = tvmService;
    }

    @Override
    public void authorize(HttpServletRequest request, AuthorizeBy handlerMethod)
            throws AuthException, UnauthorizedException
    {
        authRequestParams.setUserId(getUserIdFromBlackBox(request));
    }

    private long getUserIdFromBlackBox(HttpServletRequest request) {
        long userId;

        BlackboxCookieCredentials credentials = makeCookieCredentials(request);

        if (credentials == null) {
            throw new UnauthorizedException();
        }

        String tvmTicket = tvmIntegration.getTicket(tvmService);
        BlackboxCookieAuthRequest authRequest = new BlackboxCookieAuthRequest(credentials, tvmTicket);

        try {
            BlackboxCookieAuth blackboxCookieAuth = blackboxCookieAuthProvider.authenticate(authRequest);

            userId = blackboxCookieAuth.getUid();
        } catch (AuthenticationException e) {
            throw new UnauthorizedException();
        }

        return userId;
    }

    private BlackboxCookieCredentials makeCookieCredentials(HttpServletRequest httpServletRequest)
    {
        String host = httpServletRequest.getHeader(HttpHeaders.HOST);
        String userIp = httpServletRequest.getHeader(X_REAL_IP_HEADER);

        String sessionId = null;
        String sslSessionId = null;

        Cookie[] cookies = httpServletRequest.getCookies();

        if (cookies == null) {
            return null;
        }

        for (Cookie cookie : httpServletRequest.getCookies()) {
            String name = cookie.getName();
            if (BlackboxCookieCredentials.SESSION_ID.equals(name)) {
                sessionId = cookie.getValue();
            } else if (BlackboxCookieCredentials.SSL_SESSION_ID.equals(name)) {
                sslSessionId = cookie.getValue();
            }
        }

        return new BlackboxCookieCredentials(host, userIp, sessionId, sslSessionId);
    }

}
