package ru.yandex.canvas.configs.auth;

import javax.servlet.http.HttpServletRequest;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.core.AuthenticationException;

import ru.yandex.canvas.exceptions.AuthException;
import ru.yandex.canvas.exceptions.UnauthorizedException;
import ru.yandex.canvas.service.AuthRequestParams;
import ru.yandex.direct.common.util.HttpUtil;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;
import ru.yandex.direct.web.auth.blackbox.BlackboxOauthAuth;
import ru.yandex.direct.web.auth.blackbox.BlackboxOauthAuthProvider;
import ru.yandex.direct.web.auth.blackbox.BlackboxOauthAuthRequest;
import ru.yandex.direct.web.auth.blackbox.BlackboxOauthCredentials;

public class OAuthAuthorizer implements CanvasAuthorizer {
    private static final Logger logger = LoggerFactory.getLogger(OAuthAuthorizer.class);
    private final BlackboxOauthAuthProvider blackboxOauthAuthProvider;
    private final AuthRequestParams authRequestParams;
    private final TvmIntegration tvmIntegration;
    private final TvmService tvmService;

    public OAuthAuthorizer(BlackboxOauthAuthProvider blackboxOauthAuthProvider,
                           AuthRequestParams authRequestParams,
                           TvmIntegration tvmIntegration,
                           TvmService tvmService)
    {
        this.blackboxOauthAuthProvider = blackboxOauthAuthProvider;
        this.authRequestParams = authRequestParams;
        this.tvmIntegration = tvmIntegration;
        this.tvmService = tvmService;
    }

    private BlackboxOauthCredentials makeCredentials(HttpServletRequest request) {
        String oauthToken = HttpUtil.extractBearerToken(request, "oauth").orElse(null);
        if (oauthToken == null) {
            logger.warn("oauthToken is null");
        }
        return new BlackboxOauthCredentials(request.getServerName(), HttpUtil.getRemoteAddress(request),
                oauthToken);
    }

    @Override
    public void authorize(HttpServletRequest request, AuthorizeBy authorizeByAnnotation)
            throws AuthException, UnauthorizedException
    {
        try {
            String tvmTicket = tvmIntegration.getTicket(tvmService);
            BlackboxOauthAuth blackboxOauthAuth =
                    blackboxOauthAuthProvider.authenticate(
                            new BlackboxOauthAuthRequest(makeCredentials(request), tvmTicket));
            authRequestParams.setUserId(blackboxOauthAuth.getUid());
        } catch (AuthenticationException e) {
            logger.warn("Failed oauth ", e);
            throw new UnauthorizedException(e.getMessage());
        }
    }
}
