package ru.yandex.canvas.controllers;

import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.validation.Valid;

import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.canvas.model.CreativeDocumentBatch;
import ru.yandex.canvas.model.CreativeDocumentBatches;
import ru.yandex.canvas.model.ValidationError;
import ru.yandex.canvas.model.direct.Privileges;
import ru.yandex.canvas.model.presets.PresetSelectionCriteria;
import ru.yandex.canvas.model.presets.PresetTag;
import ru.yandex.canvas.service.AuthService;
import ru.yandex.canvas.service.CreativesService;
import ru.yandex.canvas.service.MigrationService;
import ru.yandex.canvas.service.PresetsService;
import ru.yandex.canvas.service.SessionParams;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static ru.yandex.canvas.model.presets.PresetSelectionCriteria.fromSessionParams;
import static ru.yandex.canvas.service.PresetsService.IN_BANNER_PRESET_IDS;

/**
 * @author skirsanov
 */
@RestController
@RequestMapping(value = "/creatives-batches")
public class CreativesBatchesController {
    private static final Logger logger = LoggerFactory.getLogger(CreativesBatchesController.class);

    private final CreativesService creativesService;
    private final MigrationService migrationService;
    private final SessionParams sessionParams;
    private final PresetsService presetsService;
    private final AuthService authService;


    public CreativesBatchesController(CreativesService creativesService,
                                      MigrationService migrationService,
                                      @Nonnull SessionParams sessionParams,
                                      PresetsService presetsService, AuthService authService) {
        this.creativesService = creativesService;
        this.migrationService = migrationService;
        this.sessionParams = sessionParams;
        this.presetsService = presetsService;
        this.authService = authService;
    }


    @GetMapping
    public CreativeDocumentBatches getBatches(@RequestParam("client_id") long clientId,
                                              @RequestParam(value = "offset", defaultValue = "0", required = false)
                                                      int offset,
                                              @RequestParam(value = "limit", defaultValue = "100", required = false)
                                                      int limit,
                                              @RequestParam(value = "sort_by", defaultValue = "date", required = false)
                                                      String sortBy,
                                              @RequestParam(value = "sort_order", defaultValue = "desc", required =
                                                      false)
                                                      String sortOrder,
                                              @RequestParam(value = "include", required = false)
                                                      Collection<String> include,
                                              @RequestParam(value = "exclude", defaultValue = "items.data.bundle," +
                                                      "items.data.elements,items.data.mediaSets", required = false)
                                                      Collection<String> exclude) {
        Set<Integer> presetIds = getPresetIds();

        return creativesService.getBatches(clientId, offset, limit,
                include != null ? include : emptyList(),
                exclude != null ? exclude : emptyList(),
                sortBy, "desc".equalsIgnoreCase(sortOrder), presetIds);
    }

    private PresetTag getPresetTag() {
        if (sessionParams.isPresent()) {
            switch (sessionParams.getSessionType()) {
                case CPM_GEOPRODUCT:
                    return PresetTag.CPM_GEOPRODUCT;
                case CPM_GEO_PIN:
                    return PresetTag.CPM_GEO_PIN;
            }
        }
        return PresetTag.COMMON;
    }

    /**
     * Формирует список ID шаблонов для фильтрации по тегам.
     */
    @NotNull
    Set<Integer> getPresetIds() {

        PresetTag tag = getPresetTag();

        Set<Integer> ids = new HashSet<>(presetsService.getPresetIdsByTag()
                .getOrDefault(tag, emptySet()));

        if (tag == PresetTag.COMMON) {
            ids.add(null); // All old batches don't contain presetId
        }

        //для флага инбанер=false пресеты инбанера скрываем
        if (sessionParams.getInBannerFlag() != null && !sessionParams.getInBannerFlag()) {
            ids.removeAll(IN_BANNER_PRESET_IDS);
        }

        return ids;
    }


    @GetMapping(value = "{id}")
    public CreativeDocumentBatch getBatch(@PathVariable("id") String id,
                                          @RequestParam("client_id") long clientId) {
        return creativesService.getBatch(id, clientId);
    }


    @GetMapping(value = "{id}/migrated")
    public ResponseEntity getMigratedBatch(@PathVariable("id") String id,
                                           @RequestParam("client_id") long clientId) {
        authService.requirePermission(Privileges.Permission.CREATIVE_GET);
        PresetSelectionCriteria presetSelectionCriteria = fromSessionParams(sessionParams, authService);
        return migrationService.migrate(creativesService.getBatch(id, clientId),
                presetSelectionCriteria)
                .map(creativeDocumentBatch -> (ResponseEntity) ResponseEntity.ok(creativeDocumentBatch))
                // TODO: add proper localized message
                .orElse(ResponseEntity.status(HttpStatus.PRECONDITION_FAILED).body("Can not migrate batch #" + id));
    }


    @PostMapping
    public ResponseEntity<?> createBatch(@Valid @RequestBody CreativeDocumentBatch batchDoc,
                                         BindingResult bindingResult,
                                         @RequestParam("client_id") long clientId) {
        if (bindingResult.hasErrors()) {
            logger.warn(bindingResult.getAllErrors().toString());

            return ResponseEntity.status(HttpStatus.BAD_REQUEST)
                    .contentType(MediaType.APPLICATION_JSON_UTF8)
                    .body(new ValidationError("Invalid batch creative request", bindingResult));
        }
        return ResponseEntity.status(HttpStatus.CREATED).body(creativesService.createBatch(batchDoc, clientId));
    }

    @PatchMapping(value = "{batch_id}")
    public ResponseEntity<?> updateBatchName(@RequestBody CreativeDocumentBatch batchDoc,
                                             @PathVariable("batch_id") String batchId,
                                             @RequestParam("client_id") long clientId) {
        if (StringUtils.isBlank(batchDoc.getName()) || StringUtils.isBlank(batchId)) {
            logger.warn("Update batch name error: name or id is blank");

            return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(
                    new ValidationError("Invalid batch update request"));
        }
        creativesService.updateBatchName(batchDoc, batchId, clientId);
        return ResponseEntity.status(HttpStatus.OK).build();
    }


    @DeleteMapping(value = "{id}")
    public ResponseEntity<Void> delete(@PathVariable("id") String id,
                                       @RequestParam("client_id") long clientId) {
        creativesService.delete(id, clientId);
        return ResponseEntity.status(HttpStatus.NO_CONTENT).build();
    }
}
