package ru.yandex.canvas.controllers;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import javax.validation.constraints.NotNull;

import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.canvas.model.disclaimers.Disclaimer;
import ru.yandex.canvas.model.disclaimers.DisclaimerDescription;
import ru.yandex.canvas.model.disclaimers.DisclaimerText;
import ru.yandex.canvas.service.DirectService;
import ru.yandex.canvas.service.DisclaimersService;
import ru.yandex.canvas.service.SessionParams;

import static java.util.Comparator.comparing;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.canvas.service.TankerKeySet.COUNTRIES;
import static ru.yandex.canvas.service.TankerKeySet.DISCLAIMER_GROUPS;

/**
 * @author skirsanov
 */
@RestController
@RequestMapping(value = "/disclaimers")
public class DisclaimersController {

    private final DisclaimersDTO disclaimers;

    private final DisclaimersDTO disclaimersNewFormat;

    private final SessionParams sessionParams;

    private DirectService directService;

    public DisclaimersController(final DisclaimersService disclaimersService, final DirectService directService, SessionParams sessionParams) {
        this.disclaimers = mapToDTO(disclaimersService.getList());
        this.disclaimersNewFormat = mapToDTO(disclaimersService.getListNewFormat());
        this.directService = directService;
        this.sessionParams = sessionParams;
    }

    static DisclaimersDTO mapToDTO(@NotNull final List<Disclaimer> disclaimers) {
        final List<DisclaimerGroupDTO> disclaimerGroups = disclaimers.stream()
                .map(DisclaimerGroupDTO::new)
                .sorted(comparing(DisclaimerGroupDTO::getId))
                .collect(toList());

        final List<DisclaimerDTO> items = disclaimers.stream()
                .flatMap(disclaimer -> disclaimer.getDescriptions().stream()
                        .map(description -> new DisclaimerDTO(description, disclaimer.getId())))
                // merge items with same countries
                .collect(toMap(DisclaimerDTO::getId, identity(), (a, b) -> {

                    final List<DisclaimerItemDTO> combined = new ArrayList<>(a.getItems());
                    combined.addAll(b.getItems());

                    a.setItems(combined);

                    return a;
                }))
                .values().stream()
                .sorted(comparing(DisclaimerDTO::getId))
                .collect(toList());

        return new DisclaimersDTO(disclaimerGroups, items);
    }


    @GetMapping
    public DisclaimersDTO getList() {

        Set<String> features = null;

        if (sessionParams.getClientId() != null) {
            features = directService.getFeatures(sessionParams.getClientId(), null);
        }

        if (features != null && features.contains("new_format_for_baby_food")) {
            return disclaimersNewFormat;
        }

        return disclaimers;
    }


    static class DisclaimersDTO {
        private final List<DisclaimerGroupDTO> groups;
        private final List<DisclaimerDTO> items;


        DisclaimersDTO(List<DisclaimerGroupDTO> groups, List<DisclaimerDTO> items) {
            this.groups = groups;
            this.items = items;
        }

        public List<DisclaimerGroupDTO> getGroups() {
            return groups;
        }

        public List<DisclaimerDTO> getItems() {
            return items;
        }
    }

    private static class DisclaimerGroupDTO {
        private final int id;
        private final String alias;

        DisclaimerGroupDTO(final Disclaimer disclaimer) {
            this.id = disclaimer.getId();
            this.alias = disclaimer.getFlag();
        }

        public int getId() {
            return id;
        }

        public String getAlias() {
            return alias;
        }

        public String getName() {
            return DISCLAIMER_GROUPS.key(alias);
        }
    }

    static class DisclaimerDTO {
        private final int id; //country_code
        private final String name; //country
        private List<DisclaimerItemDTO> items;


        DisclaimerDTO(final DisclaimerDescription description, final int groupId) {
            this.id = description.getCountryCode();
            this.name = description.getCountry();
            this.items = description.getTexts().stream().map(t -> new DisclaimerItemDTO(t, groupId)).collect(toList());
        }

        public int getId() {
            return id;
        }

        public String getName() {
            return COUNTRIES.key(name);
        }

        public List<DisclaimerItemDTO> getItems() {
            return items;
        }

        public void setItems(List<DisclaimerItemDTO> items) {
            this.items = items;
        }
    }

    private static class DisclaimerItemDTO {
        private final int id;
        private final String name;
        private final int groupId;

        DisclaimerItemDTO(final DisclaimerText text, int groupId) {
            this.id = text.getId();
            this.name = text.getName();
            this.groupId = groupId;
        }

        public int getId() {
            return id;
        }

        public String getName() {
            return name;
        }

        public int getGroupId() {
            return groupId;
        }
    }
}
