package ru.yandex.canvas.controllers;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.List;
import java.util.stream.IntStream;

import javax.validation.Valid;

import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RequestPart;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.client.HttpClientErrorException;
import org.springframework.web.client.HttpStatusCodeException;
import org.springframework.web.multipart.MultipartFile;

import ru.yandex.canvas.configs.auth.AuthorizeBy;
import ru.yandex.canvas.exceptions.AuthException;
import ru.yandex.canvas.exceptions.BadRequestException;
import ru.yandex.canvas.exceptions.NotFoundException;
import ru.yandex.canvas.model.CropParameters;
import ru.yandex.canvas.model.File;
import ru.yandex.canvas.model.Files;
import ru.yandex.canvas.model.ValidationError;
import ru.yandex.canvas.model.ideas.IdeaFilesHelper;
import ru.yandex.canvas.model.stock.StockFile;
import ru.yandex.canvas.model.stock.StockFiles;
import ru.yandex.canvas.service.FileService;
import ru.yandex.canvas.service.SessionParams;
import ru.yandex.canvas.service.StockService;
import ru.yandex.canvas.service.idea.IdeasService;
import ru.yandex.direct.tvm.AllowServices;

import static java.util.stream.Collectors.toList;
import static ru.yandex.canvas.configs.auth.AuthorizeBy.AuthType.BLACKBOX;
import static ru.yandex.canvas.configs.auth.AuthorizeBy.AuthType.OAUTH;
import static ru.yandex.canvas.configs.auth.AuthorizeBy.AuthType.TVM_TOKEN;
import static ru.yandex.canvas.service.AvatarsService.TURBO_NAMESPACE;
import static ru.yandex.canvas.service.TankerKeySet.ERROR;
import static ru.yandex.direct.tvm.TvmService.DIRECT_DEVELOPER;
import static ru.yandex.direct.tvm.TvmService.LPC_SWITCHCONSTRUCTOR_PROD;
import static ru.yandex.direct.tvm.TvmService.LPC_SWITCHCONSTRUCTOR_TEST;

/**
 * @author skirsanov
 */
@RestController
@RequestMapping(value = "/files")
@AuthorizeBy(value = {BLACKBOX, OAUTH, TVM_TOKEN}, tvmAllowedServices = @AllowServices(testing = {DIRECT_DEVELOPER,
        LPC_SWITCHCONSTRUCTOR_TEST}, production = {LPC_SWITCHCONSTRUCTOR_PROD}))
public class FilesController {
    private final FileService fileService;
    private final StockService stockService;
    private final IdeasService ideasService;
    private final String fileApiInternalToken;
    private final SessionParams sessionParams;

    public FilesController(final FileService fileService, final StockService stockService,
                           final IdeasService ideasService, final String fileApiInternalToken,
                           SessionParams sessionParams) {
        this.fileService = fileService;
        this.stockService = stockService;
        this.ideasService = ideasService;
        this.fileApiInternalToken = fileApiInternalToken;
        this.sessionParams = sessionParams;
    }

    @ExceptionHandler({HttpClientErrorException.class})
    public ResponseEntity handleHttpError(HttpStatusCodeException e) {
        return ResponseEntity.status(e.getStatusCode()).body(new ValidationError(ERROR.key("image-not-found")));
    }

    /**
     * @param file - multipart file content passed within "attachment" parameter due to frontend stack
     */
    @PostMapping
    public File uploadFile(@RequestParam("attachment") final MultipartFile file,
                           @RequestParam("client_id") final long clientId,
                           @RequestParam(value = "animation_allowed", required = false, defaultValue = "false") final Boolean animationAllowed,
                           @RequestParam(value = "avatars_namespace", required = false) final String avatarsNamespace) {
        if (avatarsNamespace != null && avatarsNamespace.compareTo(TURBO_NAMESPACE) != 0) {
            throw new BadRequestException("Invalid avatars namespace: " + avatarsNamespace);
        }
        File uploadedFile = fileService.uploadFile(file, null, null, clientId,
                avatarsNamespace != null, false, animationAllowed);

        if (avatarsNamespace != null) {
            fileService.turbonizeFile(uploadedFile);
        }

        return uploadedFile;
    }

    @PostMapping(value = "{originalFileId}/cropped")
    public Files uploadFilesFromCropped(@PathVariable("originalFileId") final String originalFileId,
                                        @RequestParam("attachment") final List<MultipartFile> files,
                                        // TODO: validate that maybe?
                                        // FIXME: make parameter required when front-end releases multicrop
                                        @RequestPart(value = "cropParameters", required = false) final List<CropParameters> cropParameters,
                                        @RequestParam(value = "animation_allowed", required = false, defaultValue =
                                                "false") final Boolean animationAllowed,
                                        @RequestParam("client_id") final long clientId) {
        if (cropParameters != null && cropParameters.size() != files.size()) {
            throw new BadRequestException();
        }
        return fileService.uploadFiles(files, originalFileId, cropParameters, clientId, false,
                animationAllowed, null);
    }

    // новая ручка, которая делает кроп на бэке, а не на фронте
    @PostMapping(value = "crop")
    public Files cropAndUploadFiles(@RequestParam(value = "originalFileId", required = false) final String originalFileId,
                                    @RequestParam("attachment") final List<MultipartFile> files,
                                    @RequestPart(value = "cropParameters", required = false) final List<CropParameters> cropParameters,
                                    @RequestParam(value = "animation_allowed", required = false, defaultValue =
                                            "false") final Boolean animationAllowed,
                                    @RequestParam("client_id") final long clientId) {
        if (cropParameters != null && cropParameters.size() != files.size()) {
            throw new BadRequestException();
        }
        return fileService.uploadFiles(files, originalFileId, cropParameters, clientId, true, animationAllowed, null);
    }

    @PostMapping(value = "url")
    public ResponseEntity<?> uploadFile(@RequestBody final FileUpload fileUpload,
                                        @RequestParam("client_id") final long clientId,
                                        @RequestParam(value = "avatars_namespace", required = false) final String avatarsNamespace) {
        if (avatarsNamespace != null && avatarsNamespace.compareTo(TURBO_NAMESPACE) != 0) {
            throw new BadRequestException("Invalid avatars namespace: " + avatarsNamespace);
        }
        try {
            URL fileUrl = new URL(fileUpload.getFileUrl());
            File uploadedFile = fileService.uploadFile(fileUrl, clientId);

            if (avatarsNamespace != null) {
                fileService.turbonizeFile(uploadedFile);
            }

            return ResponseEntity.ok(uploadedFile);
        } catch (MalformedURLException e) {
            return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(new ValidationError(ERROR.key("malformed-url")));
        }
    }

    @PostMapping(value = "stillage/{id}")
    public ResponseEntity<?> uploadFileByStillageId(@PathVariable("id") final String stillageFileId,
                                                    @RequestParam("client_id") final long clientId) {
        return ResponseEntity.ok(fileService.uploadFile(stillageFileId, clientId));
    }

    @GetMapping(value = "stock")
    public StockFiles getStockFiles(@RequestParam("client_id") final long clientId,
                                    @RequestParam(value = "category_id", required = false) final int[] categoryIds,
                                    @RequestParam(value = "limit", defaultValue = "24") int limit,
                                    @RequestParam(value = "offset", defaultValue = "0") int offset) {
        if (limit < 0 || offset < 0) {
            throw new BadRequestException();
        }

        if (categoryIds == null) {
            return stockService.getStockFiles(limit, offset);
        } else {
            return stockService.getStockFiles(IntStream.of(categoryIds).boxed().collect(toList()), limit, offset);
        }
    }

    @GetMapping(value = "idea/{ideaId}")
    public ResponseEntity<Files> getIdeaFiles(@RequestParam("client_id") final long clientId,
                                              @PathVariable("ideaId") String ideaId) {
        return ideasService.getIdea(ideaId, clientId)
                .map(idea -> ResponseEntity.ok(IdeaFilesHelper.extractFiles(idea)))
                .orElse(ResponseEntity.status(HttpStatus.NOT_FOUND).build());
    }

    @PostMapping(value = "idea/{ideaId}/file/{ideaFileId}")
    public ResponseEntity<File> uploadFile(@RequestParam("client_id") long clientId,
                                           @PathVariable("ideaId") String ideaId,
                                           @PathVariable("ideaFileId") String ideaFileId) {
        return ideasService.getIdea(ideaId, clientId)
                .map(ideaDocument -> ideaDocument.getScraperData().getImages().stream()
                        .filter(file -> ideaFileId.equals(file.getId()))
                        .findAny()
                        .map(requestedFile -> ResponseEntity.status(HttpStatus.CREATED)
                                .body(fileService.uploadIdeaFile(requestedFile.getId(), ideaId, clientId)))
                        .orElse(ResponseEntity.status(HttpStatus.NOT_FOUND).build()))
                .orElseGet(() -> ResponseEntity.status(HttpStatus.NOT_FOUND).build());
    }

    @PostMapping(value = "from-stock/{stockFileId}/used")
    public ResponseEntity<File> uploadFile(@PathVariable("stockFileId") final String stockFileId,
                                           @RequestParam("client_id") final long clientId) {
        final StockFile stockFile = stockService.getFile(stockFileId).orElseThrow(NotFoundException::new);
        stockFile.setId(stockFileId);
        return ResponseEntity.status(HttpStatus.CREATED).body(fileService.saveFileFromStock(stockFile, clientId));
    }

    @GetMapping(value = "{id}")
    public File getById(@PathVariable("id") final String id,
                        @RequestParam("client_id") final long clientId) {
        return fileService.getById(id, clientId).orElseThrow(NotFoundException::new);
    }

    @GetMapping(value = "{id}/internal")
    public File getByIdInternal(@PathVariable("id") final String id,
                                @RequestHeader("Authorization") final String token) {
        if (!token.equalsIgnoreCase(this.fileApiInternalToken)) {
            throw new AuthException("invalid token");
        }

        return fileService.getByIdInternal(id).orElseThrow(NotFoundException::new);
    }

    @GetMapping(value = "my")
    public Files getMyFiles(@RequestParam(value = "name", required = false) String name,
                            @RequestParam(value = "offset", defaultValue = "0") int offset,
                            @RequestParam(value = "limit", defaultValue = "24") int limit,
                            @RequestParam(value = "sort_order", defaultValue = "desc") final String sortOrder,
                            @RequestParam(value = "show_feeds", required = false) final Boolean showFeeds,
                            @RequestParam("client_id") final long clientId) {
        if (limit < 0 || offset < 0) {
            throw new BadRequestException();
        }
        return fileService.find(clientId, name, "desc".equalsIgnoreCase(sortOrder), offset, limit, showFeeds);
    }

    @PutMapping(value = "{id}")
    public File updateFile(@PathVariable("id") final String id,
                           @RequestParam("client_id") final long clientId,
                           @Valid @RequestBody final File file) {
        file.setId(id);
        return fileService.update(file, clientId);
    }

    @DeleteMapping(value = "{id}")
    public ResponseEntity<Void> deleteFile(@PathVariable("id") final String id,
                                           @RequestParam("client_id") final long clientId) {
        fileService.delete(id, clientId);
        return ResponseEntity.status(HttpStatus.NO_CONTENT).build();
    }

    private static class FileUpload {
        private String fileUrl;

        public String getFileUrl() {
            return fileUrl;
        }

        public void setFileUrl(String fileUrl) {
            this.fileUrl = fileUrl;
        }

        @Override
        public String toString() {
            return "FileUpload{" +
                    "fileUrl='" + fileUrl + '\'' +
                    '}';
        }
    }
}
