package ru.yandex.canvas.controllers;

import javax.validation.Valid;

import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.canvas.model.DraftCreativeBatch;
import ru.yandex.canvas.model.IdeaDocument;
import ru.yandex.canvas.model.direct.Privileges;
import ru.yandex.canvas.model.ideas.IdeaRequest;
import ru.yandex.canvas.model.presets.PresetSelectionCriteria;
import ru.yandex.canvas.model.scraper.ScraperData;
import ru.yandex.canvas.service.AuthService;
import ru.yandex.canvas.service.SessionParams;
import ru.yandex.canvas.service.idea.IdeasService;
import ru.yandex.canvas.service.scraper.ScraperService;

import static ru.yandex.canvas.model.presets.PresetSelectionCriteria.fromSessionParams;

@RestController
@RequestMapping(value = "/ideas")
public class IdeasController {

    private final IdeasService ideasService;
    private final ScraperService scraperService;
    private final SessionParams sessionParams;
    private final AuthService authService;

    public IdeasController(IdeasService ideaService, ScraperService scraperService, SessionParams sessionParams,
                           AuthService authService) {
        this.ideasService = ideaService;
        this.scraperService = scraperService;
        this.sessionParams = sessionParams;
        this.authService = authService;
    }

    @PostMapping
    ResponseEntity<DraftCreativeBatch> createIdea(@Valid @RequestBody final IdeaRequest ideaRequest,
                                                  @RequestParam("client_id") long clientId) {
        ScraperData scraperData = scraperService.scrapSiteInfo(ideaRequest.getUrl());
        IdeaDocument idea = ideasService.createIdea(scraperData, ideaRequest.getUrl(), clientId);
        PresetSelectionCriteria selectionCriteria = fromSessionParams(sessionParams, authService);

        return ResponseEntity.status(HttpStatus.CREATED).
                body(new DraftCreativeBatch(ideasService.previewDraftCreatives(idea, selectionCriteria), idea));
    }

    @GetMapping(value = "{id}")
    ResponseEntity<DraftCreativeBatch> getIdea(@PathVariable("id") String ideaId,
                                               @RequestParam("client_id") long clientId) {
        PresetSelectionCriteria selectionCriteria = fromSessionParams(sessionParams, authService);
        return ideasService.getIdea(ideaId, clientId)
                .map(idea -> ResponseEntity
                        .ok(new DraftCreativeBatch(ideasService.previewDraftCreatives(idea, selectionCriteria), idea)))
                .orElse(ResponseEntity.status(HttpStatus.NOT_FOUND).build());
    }

    @GetMapping(value = "{ideaId}/presets/{presetId}")
    ResponseEntity<DraftCreativeBatch> generateCreatives(@PathVariable("ideaId") String ideaId,
                                                         @PathVariable("presetId") int presetId,
                                                         @RequestParam("client_id") long clientId) {
        authService.requirePermission(Privileges.Permission.CREATIVE_GET);
        PresetSelectionCriteria selectionCriteria = fromSessionParams(sessionParams, authService);
        return ideasService.getIdea(ideaId, clientId)
                .map(idea -> ResponseEntity.ok(new DraftCreativeBatch(ideasService
                        .generatePresetDraftCreatives(idea, presetId, selectionCriteria), idea)))
                .orElse(ResponseEntity.status(HttpStatus.NOT_FOUND).build());
    }
}
