package ru.yandex.canvas.controllers;

import java.io.IOException;
import java.net.URISyntaxException;
import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.validation.BindingResult;
import org.springframework.validation.Validator;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.canvas.controllers.html5.Html5CreativeController;
import ru.yandex.canvas.exceptions.InternalServerError;
import ru.yandex.canvas.exceptions.NotFoundException;
import ru.yandex.canvas.model.CreativeDocument;
import ru.yandex.canvas.model.MediaSetSubItem;
import ru.yandex.canvas.model.ValidationError;
import ru.yandex.canvas.model.direct.Privileges;
import ru.yandex.canvas.model.html_builder.InBannerHtmlCreativeWriter;
import ru.yandex.canvas.service.AuthService;
import ru.yandex.canvas.service.CreativesService;
import ru.yandex.canvas.service.SessionParams;
import ru.yandex.canvas.service.html5.Preview;
import ru.yandex.canvas.service.video.InBannerVideoFilesService;

@RestController
@RequestMapping(value = "/creatives/in_banner")
public class InBannerPreviewController {
    private static final Logger logger = LoggerFactory.getLogger(InBannerPreviewController.class);

    private final InBannerHtmlCreativeWriter inBannerHtmlCreativeWriter;
    private final AuthService authService;
    private final CreativesService creativesService;
    private final SessionParams sessionParams;
    private final Validator validator;
    private final InBannerVideoFilesService inBannerVideoFilesService;

    public InBannerPreviewController(InBannerHtmlCreativeWriter inBannerHtmlCreativeWriter, AuthService authService,
                                     CreativesService creativesService,
                                     SessionParams sessionParams, Validator validator,
                                     InBannerVideoFilesService inBannerVideoFilesService) {
        this.inBannerHtmlCreativeWriter = inBannerHtmlCreativeWriter;
        this.authService = authService;
        this.creativesService = creativesService;
        this.sessionParams = sessionParams;
        this.validator = validator;
        this.inBannerVideoFilesService = inBannerVideoFilesService;
    }

    @GetMapping(value = "{batch_id}/{id}/preview")
    public ResponseEntity<?> getPreviewInfoById(@PathVariable("batch_id") String batchId,
                                                @PathVariable("id") long id,
                                                @RequestParam(value = "nonce", required = false) String nonce) {
        authService.requirePermission(Privileges.Permission.PREVIEW);

        Optional<CreativeDocument> document = creativesService.getCreativeInternalForPreview(id, batchId);

        if (document.isPresent()) {
            try {
                return ResponseEntity
                        .status(HttpStatus.OK)
                        .contentType(MediaType.TEXT_HTML)
                        .body(makePreviewHtml(document.get(), Preview.DataParams.defaultDataParams(),
                                sessionParams.getClientId(), nonce));
            } catch (IOException | URISyntaxException e) {
                throw new InternalServerError(e);
            }
        } else {
            throw new NotFoundException();
        }
    }

    private String makePreviewHtml(CreativeDocument creativeDocument, Preview.DataParams urlData, Long clientId,
                                   String nonce) throws IOException, URISyntaxException {
        return inBannerHtmlCreativeWriter.composeCreativeHTML(creativeDocument, clientId, urlData.toJson(), nonce);
    }

    @RequestMapping(method = RequestMethod.POST)
    public ResponseEntity<?> createPreviewFromCreativeDocument(@RequestBody CreativeDocument creativeDocument,
                                                               BindingResult bindingResult,
                                                               @RequestParam(value = "nonce", required = false) String nonce,
                                                               @RequestParam("client_id") long clientId) throws IOException, URISyntaxException {
        authService.requirePermission(Privileges.Permission.CREATIVE_CREATE);

        MediaSetSubItem videoElement = inBannerVideoFilesService.getVideoMediaItem(creativeDocument);

        if (videoElement != null && videoElement.getFileId() == null) {
            videoElement.setDefaultValue(true);
            videoElement.setFileId("");
        }

        MediaSetSubItem imageElement = inBannerVideoFilesService.getImageMediaItem(creativeDocument);

        if (imageElement != null && imageElement.getEffectiveFileId() == null) {
            imageElement.setDefaultValue(true);
            imageElement.setFileId("");
        }

        MediaSetSubItem secondImageElement = inBannerVideoFilesService.getSecondImageMediaItem(creativeDocument);

        if (secondImageElement != null && secondImageElement.getEffectiveFileId() == null) {
            secondImageElement.setDefaultValue(true);
            secondImageElement.setFileId("");
        }

        validator.validate(creativeDocument, bindingResult);

        if (bindingResult.hasErrors()) {
            logger.warn(bindingResult.getAllErrors().toString());

            return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(
                    new ValidationError("Invalid batch creative request", bindingResult));
        }

        return ResponseEntity
                .status(HttpStatus.OK)
                .body(new Html5CreativeController.Response().setContent(makePreviewHtml(creativeDocument,
                        Preview.DataParams.defaultDataParams(), clientId, nonce))
                        .setHeight(creativeDocument.getData().getHeight())
                        .setWidth(creativeDocument.getData().getWidth())
                );

    }

}
