package ru.yandex.canvas.controllers;

import java.io.IOException;
import java.math.BigInteger;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.annotation.JsonProperty;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;

import ru.yandex.canvas.configs.auth.AuthorizeBy;
import ru.yandex.canvas.model.CreativeWithId;
import ru.yandex.canvas.model.video.VideoFiles;
import ru.yandex.canvas.model.video.files.FileStatus;
import ru.yandex.canvas.model.video.vc.files.VideoConstructorFile;
import ru.yandex.canvas.repository.video.VideoAdditionsRepository;
import ru.yandex.canvas.repository.video.VideoFilesRepository;
import ru.yandex.canvas.service.OnCreativeService;
import ru.yandex.canvas.service.multitype.CreativeType;
import ru.yandex.canvas.service.multitype.OnCreativeOperationResult;
import ru.yandex.canvas.service.multitype.OnCreativeServiceProvider;
import ru.yandex.canvas.service.multitype.operation.AdminRejectOperation;
import ru.yandex.canvas.service.multitype.operation.RebuildOperation;
import ru.yandex.canvas.service.multitype.operation.ScreenshotOperation;
import ru.yandex.canvas.service.multitype.operation.SendToDirectOperation;
import ru.yandex.canvas.service.multitype.operation.SendToRtbHostOperation;
import ru.yandex.canvas.service.multitype.request.AdminRejectOperationMultiTypeRequest;
import ru.yandex.canvas.service.multitype.request.DefaultRequest;
import ru.yandex.canvas.service.video.VhService;
import ru.yandex.canvas.service.video.VideoAdditionsService;
import ru.yandex.canvas.service.video.VideoConstructorFilesService;
import ru.yandex.canvas.service.video.VideoFileUploadServiceInterface;
import ru.yandex.direct.canvas.CanvasToolsRequests;
import ru.yandex.direct.tvm.AllowServices;

import static ru.yandex.direct.tvm.TvmService.DIRECT_AUTOTESTS;
import static ru.yandex.direct.tvm.TvmService.DIRECT_CANVAS_PROD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_CANVAS_TEST;
import static ru.yandex.direct.tvm.TvmService.DIRECT_DEVELOPER;
import static ru.yandex.direct.tvm.TvmService.DIRECT_SCRIPTS_PROD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_SCRIPTS_TEST;
import static ru.yandex.direct.tvm.TvmService.DIRECT_WEB_PROD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_WEB_TEST;
import static ru.yandex.direct.utils.StringUtils.nullIfBlank;

/**
 * Контроллер для внутренних сервисных операций. Его же пердполагается дергать из внутренних отчетов Директа
 * т.к. собственного админского интерфейса у Канваса нет
 * Для удобства работает со всеми типами креативов (все трех канвасов)
 */

/**
 * * Добавление новой операции не должно влиять на предыдущие
 * * Добавление нового типа креативов не должно влиять на уже существующие операции (они должны работать
 * по исходномым типам)
 * * Должно быть легко писать операции работающие по разным ограниченным наборам типов
 * * Логика работы с типом должна быть инкапсулирована, чтобы не приходилось в одном классе работать, например,
 * с сервисами разных типов креативов, т.к. будет пересечение имен.
 */

@ParametersAreNonnullByDefault
@RestController
@Lazy
@RequestMapping(value = "/tools")
@AuthorizeBy(value = {AuthorizeBy.AuthType.TVM_TOKEN}, tvmAllowedServices = @AllowServices(testing = {DIRECT_AUTOTESTS,
        DIRECT_DEVELOPER, DIRECT_CANVAS_TEST, DIRECT_SCRIPTS_TEST, DIRECT_WEB_TEST}, production =
        {DIRECT_CANVAS_PROD, DIRECT_SCRIPTS_PROD, DIRECT_WEB_PROD}))
public class InternalToolsController {
    private final ScreenshotOperation screenshotOperation;
    private final SendToDirectOperation sendToDirectOperation;
    private final SendToRtbHostOperation sendToRtbHostOperation;
    private final RebuildOperation rebuildOperation;
    private final AdminRejectOperation adminRejectOperation;
    private final OnCreativeServiceProvider onCreativeServiceProvider;
    private final VideoFileUploadServiceInterface videoFileUploadService;
    private final VideoFilesRepository videoFilesRepository;
    private final VideoAdditionsService videoAdditionsService;
    private final VideoAdditionsRepository videoAdditionsRepository;
    private final VideoConstructorFilesService videoConstructorFilesService;
    private final VhService vhService;

    public InternalToolsController(ScreenshotOperation screenshotOperation,
                                   SendToDirectOperation sendToDirectOperation,
                                   SendToRtbHostOperation sendToRtbHostOperation,
                                   RebuildOperation rebuildOperation,
                                   AdminRejectOperation adminRejectOperation,
                                   OnCreativeServiceProvider onCreativeServiceProvider,
                                   VideoFileUploadServiceInterface videoFileUploadService,
                                   VideoFilesRepository videoFilesRepository,
                                   VideoAdditionsService videoAdditionsService,
                                   VideoAdditionsRepository videoAdditionsRepository, VideoConstructorFilesService videoConstructorFilesService, VhService vhService) {
        this.screenshotOperation = screenshotOperation;
        this.sendToDirectOperation = sendToDirectOperation;
        this.sendToRtbHostOperation = sendToRtbHostOperation;
        this.rebuildOperation = rebuildOperation;
        this.adminRejectOperation = adminRejectOperation;
        this.onCreativeServiceProvider = onCreativeServiceProvider;
        this.videoFileUploadService = videoFileUploadService;
        this.videoFilesRepository = videoFilesRepository;
        this.videoAdditionsService = videoAdditionsService;
        this.videoAdditionsRepository = videoAdditionsRepository;
        this.videoConstructorFilesService = videoConstructorFilesService;
        this.vhService = vhService;
    }

    /**
     * @param creativeIds
     * @return map id->result, если значение из запроса отсутствует в ответе, значит креатив с таким id не найден
     */
    @PostMapping(value = "reshoot_screenshot")
    public ResponseEntity<Map<Long, OnCreativeOperationResult>> reshootScreenshot(@RequestBody List<Long> creativeIds) {
        return ResponseEntity.ok(screenshotOperation.run(DefaultRequest.of(creativeIds)));
    }

    @PostMapping(value = "rebuild")
    public ResponseEntity<Map<Long, OnCreativeOperationResult>> rebuild(@RequestBody List<Long> creativeIds) {
        return ResponseEntity.ok(rebuildOperation.run(DefaultRequest.of(creativeIds)));
    }

    @PostMapping(value = "send_to_direct")
    public ResponseEntity<Map<Long, OnCreativeOperationResult>> sendToDirect(@RequestBody List<Long> creativeIds) {
        return ResponseEntity.ok(sendToDirectOperation.run(DefaultRequest.of(creativeIds)));
    }

    @PostMapping(value = "send_to_rtbhost")
    public ResponseEntity<Map<Long, OnCreativeOperationResult>> sendToRtbhost(@RequestBody List<Long> creativeIds) {
        return ResponseEntity.ok(sendToRtbHostOperation.run(DefaultRequest.of(creativeIds)));
    }

    @PostMapping(value = "admin_reject_creatives")
    public ResponseEntity<Map<Long, OnCreativeOperationResult>> adminRejectCreatives(
            @RequestBody CanvasToolsRequests.AdminRejectRequest request) {
        return ResponseEntity.ok(adminRejectOperation.run(
                AdminRejectOperationMultiTypeRequest.of(request.getCreativeIdsList(),
                        nullIfBlank(request.getReason()))));
    }

    @GetMapping(value = "video-vast/{id}")
    public ResponseEntity getVideoVast(@PathVariable("id") Long creativeId) {
        var addition = videoAdditionsService.getAdditionByCreativeIdArchivedAlso(creativeId);

        if (addition == null) {
            return ResponseEntity.notFound().build();
        }

        return ResponseEntity.ok().contentType(MediaType.APPLICATION_XML).body(addition.getVast());
    }

    //        videoAdditionsRepository.updateVastById(addition.getId(), vast);
    @PostMapping(value = "video-vast/{id}")
    public ResponseEntity patchVideoVast(@PathVariable("id") Long creativeId,
                                         @RequestParam("file") final MultipartFile file) throws IOException {
        var addition = videoAdditionsService.getAdditionByCreativeIdArchivedAlso(creativeId);

        if (addition == null) {
            return ResponseEntity.notFound().build();
        }

        videoAdditionsRepository.updateVastById(addition.getId(), new String(file.getBytes(),
                StandardCharsets.UTF_8));

        addition = videoAdditionsService.getAdditionByCreativeIdArchivedAlso(creativeId);
        return ResponseEntity.ok(addition.getVast());
    }

    @GetMapping(value = "creative/{id}")
    public ResponseEntity getCreativeData(@PathVariable("id") Long creativeId) {
        for (CreativeType value : CreativeType.values()) {
            OnCreativeService service = onCreativeServiceProvider.provide(value);
            List<CreativeWithId> list = service.fetchByIds(Collections.singletonList(creativeId));
            if (!list.isEmpty()) {
                return ResponseEntity.ok(new TypeWithCreative(value, list.get(0)));
            }
        }
        return ResponseEntity.notFound().build();
    }

    @PostMapping(path = "/regenerate-video/{id}")
    public ResponseEntity<Long> regenerateVideo(@PathVariable("id") String fileId) {
        VideoFiles file = videoFilesRepository.findById(fileId);
        Long taskId = videoFileUploadService.startSandboxConverting(file, false);
        videoFilesRepository.update(fileId, new VideoFilesRepository.UpdateBuilder()
                .withStatus(FileStatus.CONVERTING)
                .withConversionId(taskId));
        return ResponseEntity.ok(taskId);
    }

    @PostMapping(path = "/cms-recode/{id}")
    public ResponseEntity<String> cmsRecode(@PathVariable("id") String fileId) {
        VideoFiles file = videoFilesRepository.findById(fileId);
        BigInteger videoMetaId = vhService.startEncoding(file.getUrl()).getVideoMetaId();
        videoFilesRepository.update(fileId, new VideoFilesRepository.UpdateBuilder()
                .withStatus(FileStatus.CONVERTING)
                .withConversionId(null)
                .withVideoMetaId(videoMetaId)
        );
        return ResponseEntity.ok(String.valueOf(videoMetaId));
    }

    @PostMapping(path = "/constructor-files/rebuild_packshot")
    public ResponseEntity<Long> rebuildPackshot(@RequestBody List<String> fileIds) {
        List<VideoConstructorFile> videoConstructorFiles = videoConstructorFilesService.getByIdsInternal(fileIds);
        Long taskId = videoConstructorFilesService.startRebuild(videoConstructorFiles,
                VideoConstructorFilesService.SandboxTask.REBUILD_PACKSHOT);
        return ResponseEntity.ok(taskId);
    }

    @PostMapping(path = "/constructor-files/rebuild_preview")
    public ResponseEntity<Long> rebuildPreview(@RequestBody List<String> fileIds) {
        List<VideoConstructorFile> videoConstructorFiles = videoConstructorFilesService.getByIdsInternal(fileIds);
        Long taskId = videoConstructorFilesService.startRebuild(videoConstructorFiles,
                VideoConstructorFilesService.SandboxTask.REBUILD_PREVIEW);
        return ResponseEntity.ok(taskId);
    }

    private class TypeWithCreative {
        @JsonProperty("type")
        private CreativeType type;

        @JsonProperty("creative")
        private CreativeWithId creative;

        TypeWithCreative(CreativeType type, CreativeWithId creative) {
            this.type = type;
            this.creative = creative;
        }

        public CreativeType getType() {
            return type;
        }

        public CreativeWithId getCreative() {
            return creative;
        }
    }
}
