package ru.yandex.canvas.controllers;

import java.net.MalformedURLException;
import java.net.URL;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import com.google.gson.Gson;
import com.mongodb.client.result.UpdateResult;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.mongodb.core.MongoOperations;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.canvas.configs.auth.AuthorizeBy;
import ru.yandex.canvas.model.html5.Batch;
import ru.yandex.canvas.model.html5.Creative;
import ru.yandex.canvas.model.stillage.StillageFileInfo;
import ru.yandex.canvas.repository.html5.BatchesRepository;
import ru.yandex.canvas.service.StillageService;
import ru.yandex.direct.tvm.AllowServices;

import static ru.yandex.direct.tvm.TvmService.DIRECT_AUTOTESTS;
import static ru.yandex.direct.tvm.TvmService.DIRECT_CANVAS_PROD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_CANVAS_TEST;
import static ru.yandex.direct.tvm.TvmService.DIRECT_DEVELOPER;
import static ru.yandex.direct.tvm.TvmService.DIRECT_SCRIPTS_PROD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_SCRIPTS_TEST;
import static ru.yandex.direct.tvm.TvmService.DIRECT_WEB_PROD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_WEB_TEST;

/**
 * @author lightelfik
 */


@AuthorizeBy(value = {AuthorizeBy.AuthType.TVM_TOKEN}, tvmAllowedServices = @AllowServices(
        testing = {DIRECT_AUTOTESTS, DIRECT_DEVELOPER, DIRECT_CANVAS_TEST, DIRECT_SCRIPTS_TEST, DIRECT_WEB_TEST},
        production = {DIRECT_CANVAS_PROD, DIRECT_SCRIPTS_PROD, DIRECT_WEB_PROD, DIRECT_DEVELOPER}))
@RestController
@RequestMapping(value = "/wrongJsSources")
public class JsReloadController {
    private static final Logger logger = LoggerFactory.getLogger(JsReloadController.class);
    private static final DateTime DATE_TIME = new DateTime().withDate(2019, 7, 25);

    private final StillageService stillageService;
    private final MongoOperations mongoOperations;

    public JsReloadController(MongoOperations mongoOperations, StillageService stillageService) {
        this.stillageService = stillageService;
        this.mongoOperations = mongoOperations;
    }

    @GetMapping("/get")
    public Map<String, String> getReplacements(@RequestParam(name = "client_id", required = false) Long clientId) {

        List<Batch> batches;
        if (clientId != null) {
            batches = new BatchesRepository(mongoOperations).getBatchesWithHtmlReplacementsByClientId(clientId,
                    DATE_TIME);
        } else {
            batches = new BatchesRepository(mongoOperations).getBatchesWithHtmlReplacements(DATE_TIME);
        }

        Map<String, String> mdsUrls = batches.stream()
                .map(b -> b.getCreatives())
                .flatMap(c -> c.stream())
                .map(c -> c.getSource().getHtmlReplacements())
                .flatMap(r -> r.stream())
                .collect(Collectors.toMap(r -> r.get(1), r -> r.get(0), (url1, url2) -> {
                    return url1;
                }));

        return mdsUrls;
    }

    @PostMapping("/upload")
    public Map<String, String> uploadHtml5Sources(@RequestParam(name = "client_id", required = false) Long clientId) {

        List<Batch> batches;
        if (clientId != null) {
            batches = new BatchesRepository(mongoOperations).getBatchesWithHtmlReplacementsByClientId(clientId,
                    DATE_TIME);
        } else {
            batches = new BatchesRepository(mongoOperations).getBatchesWithHtmlReplacements(DATE_TIME);
        }

        Map<String, String> mdsUrls = batches.stream()
                .map(b -> b.getCreatives())
                .flatMap(c -> c.stream())
                .map(c -> c.getSource().getHtmlReplacements())
                .flatMap(r -> r.stream())
                .collect(Collectors.toMap(r -> r.get(1), r -> r.get(0), (url1, url2) -> {
                    return url1;
                }));

        Map<String, String> oldUrlToNew = reuploadAndReturnOldToNewUrls(mdsUrls);

        return oldUrlToNew;
    }

    private Map<String, String> reuploadAndReturnOldToNewUrls(Map<String, String> mdsUrls) {
        Map<String, String> oldUrlToNew = new HashMap<>();

        for (String url : mdsUrls.keySet()) {
            if (!oldUrlToNew.containsKey(url)) {
                try {
                    URL originalUrl = new URL(mdsUrls.get(url));
                    String libName = Paths.get(originalUrl.getPath()).getFileName().toString();
                    URL mdsUrl = new URL(url);
                    StillageFileInfo fileInfo = stillageService.uploadFile(libName, mdsUrl);

                    oldUrlToNew.put(url, fileInfo.getUrl());
                    logger.warn("Url " + url + ", NewUrl " + fileInfo.getUrl());
                } catch (MalformedURLException e) {
                    logger.warn("Invalid url " + url, e);
                }
            }
        }
        return oldUrlToNew;
    }

    @PostMapping("/update")
    public Map<String, String> updateHtml5Sources(
            @RequestBody String urls,
            @RequestParam(name = "do_update", required = false, defaultValue = "false") Boolean doUpdate
    ) {
        Gson gson = new Gson();

        Map<String, String> oldUrlToNew = gson.fromJson(urls, HashMap.class);

        List<Batch> batches = new BatchesRepository(mongoOperations).getBatchesWithHtmlReplacements(DATE_TIME);

        Map<String, String> replacementsByResultId = new HashMap<>();
        for (Batch batch : batches) {
            for (Creative creative : batch.getCreatives()) {
                List<List<String>> htmlReplacements = creative.getSource().getHtmlReplacements();
                if (!htmlReplacements.isEmpty()) {
                    List<List<String>> newHtmlReplacements = htmlReplacements.stream().map(
                            i -> new ArrayList<>(
                                    Arrays.asList(i.get(0), oldUrlToNew.getOrDefault(i.get(1), i.get(1)))
                            )
                    ).collect(Collectors.toList());

                    if (doUpdate) {
                        UpdateResult result = new BatchesRepository(mongoOperations)
                                .updateBatchHtmlReplacements(creative.getId(), newHtmlReplacements);
                        if (result.getModifiedCount() >= 1) {
                            replacementsByResultId.put(creative.getId().toString(), gson.toJson(newHtmlReplacements));
                            logger.warn(creative.getId() + " ModifiedCount:" + result.getModifiedCount());
                        } else {
                            logger.error(creative.getId() + " ModifiedCount:" + result.getModifiedCount());
                        }

                    } else {
                        replacementsByResultId.put(creative.getId().toString(), gson.toJson(newHtmlReplacements));
                    }

                }
            }
        }
        return replacementsByResultId;
    }
}
