package ru.yandex.canvas.controllers;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.List;
import java.util.stream.IntStream;

import javax.validation.Valid;

import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RequestPart;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.client.HttpClientErrorException;
import org.springframework.web.client.HttpStatusCodeException;
import org.springframework.web.multipart.MultipartFile;

import ru.yandex.canvas.exceptions.BadRequestException;
import ru.yandex.canvas.exceptions.NotFoundException;
import ru.yandex.canvas.model.CropParameters;
import ru.yandex.canvas.model.File;
import ru.yandex.canvas.model.Files;
import ru.yandex.canvas.model.ValidationError;
import ru.yandex.canvas.model.stock.StockFile;
import ru.yandex.canvas.model.stock.StockFiles;
import ru.yandex.canvas.service.AuthService;
import ru.yandex.canvas.service.UserFileService;
import ru.yandex.canvas.service.UserStockService;

import static java.util.stream.Collectors.toList;
import static ru.yandex.canvas.service.AvatarsService.TURBO_NAMESPACE;
import static ru.yandex.canvas.service.TankerKeySet.ERROR;

/**
 * Almost the same as FilesController, but for userId instead of clientId.
 * <p>
 * Temporary solution™ for ~1 year (starting from 2018/02/14).
 */
@RestController
@RequestMapping(value = "/user-files")
public class UserFilesController {
    private final UserFileService userFileService;
    private final UserStockService userStockService;
    private final AuthService authService;

    public UserFilesController(final UserFileService userFileService, final UserStockService userStockService,
            AuthService authService)
    {
        this.userFileService = userFileService;
        this.userStockService = userStockService;
        this.authService = authService;
    }

    @ExceptionHandler({HttpClientErrorException.class})
    public ResponseEntity handleHttpError(HttpStatusCodeException e) {
        return ResponseEntity.status(e.getStatusCode()).body(new ValidationError(ERROR.key("image-not-found")));
    }

    /**
     * @param file - multipart file content passed within "attachment" parameter due to frontend stack
     */
    @PostMapping
    public File uploadFile(@RequestParam("attachment") final MultipartFile file,
                           @RequestParam(value = "avatars_namespace", required = false) final String avatarsNamespace) {
        if (avatarsNamespace != null && avatarsNamespace.compareTo(TURBO_NAMESPACE) != 0) {
            throw new BadRequestException("Invalid avatars namespace: " + avatarsNamespace);
        }
        File uploadedFile = userFileService.uploadFile(null, file, null, authService.getUserId(), avatarsNamespace != null);

        if (avatarsNamespace != null) {
            userFileService.turbonizeFile(uploadedFile);
        }

        return uploadedFile;
    }

    @PostMapping(value = "{originalFileId}/cropped")
    public Files uploadFilesFromCropped(@PathVariable("originalFileId") final String originalFileId,
                                        @RequestParam("attachment") final List<MultipartFile> files,
                                        // TODO: validate that maybe?
                                        // FIXME: make parameter required when front-end releases multicrop
            @RequestPart(value = "cropParameters", required = false) final List<CropParameters> cropParameters)
    {
        if (cropParameters != null && cropParameters.size() != files.size()) {
            throw new BadRequestException();
        }
        return userFileService.uploadFiles(originalFileId, files, cropParameters, authService.getUserId());
    }

    @PostMapping(value = "url")
    public ResponseEntity<?> uploadFile(@RequestBody final FileUpload fileUpload) {
        try {
            URL fileUrl = new URL(fileUpload.getFileUrl());
            return ResponseEntity.ok(userFileService.uploadFile(fileUrl, authService.getUserId()));
        } catch (MalformedURLException e) {
            return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(new ValidationError(ERROR.key("malformed-url")));
        }
    }

    @GetMapping(value = "stock")
    public StockFiles getStockFiles(@RequestParam(value = "category_id", required = false) final int[] categoryIds,
                                    @RequestParam(value = "limit", defaultValue = "24") int limit,
                                    @RequestParam(value = "offset", defaultValue = "0") int offset) {
        if (limit < 0 || offset < 0) {
            throw new BadRequestException();
        }

        if (categoryIds == null) {
            return userStockService.getStockFiles(limit, offset);
        } else {
            return userStockService.getStockFiles(IntStream.of(categoryIds).boxed().collect(toList()), limit, offset);
        }
    }

    @PostMapping(value = "from-stock/{stockFileId}/used")
    public ResponseEntity<File> uploadFile(@PathVariable("stockFileId") final String stockFileId) {
        final StockFile stockFile = userStockService.getFile(stockFileId).orElseThrow(NotFoundException::new);
        stockFile.setId(stockFileId);
        return ResponseEntity.status(HttpStatus.CREATED).body(userFileService.saveFileFromStock(stockFile, authService.getUserId()));
    }

    @GetMapping(value = "{id}")
    public File getById(@PathVariable("id") final String id) {
        return userFileService.getById(id, authService.getUserId()).orElseThrow(NotFoundException::new);
    }

    @GetMapping(value = "my")
    public Files getMyFiles(@RequestParam(value = "name", required = false) String name,
                            @RequestParam(value = "offset", defaultValue = "0") int offset,
                            @RequestParam(value = "limit", defaultValue = "24") int limit,
                            @RequestParam(value = "sort_order", defaultValue = "desc", required = false) final String sortOrder) {
        if (limit < 0 || offset < 0) {
            throw new BadRequestException();
        }
        return userFileService.find(authService.getUserId(), name, "desc".equalsIgnoreCase(sortOrder), offset, limit);
    }

    @PutMapping(value = "{id}")
    public File updateFile(@PathVariable("id") final String id,
                           @Valid @RequestBody final File file) {
        file.setId(id);
        return userFileService.update(file, authService.getUserId());
    }

    @DeleteMapping(value = "{id}")
    public ResponseEntity<Void> deleteFile(@PathVariable("id") final String id) {
        userFileService.delete(id, authService.getUserId());
        return ResponseEntity.status(HttpStatus.NO_CONTENT).build();
    }

    private static class FileUpload {
        private String fileUrl;

        public String getFileUrl() {
            return fileUrl;
        }

        public void setFileUrl(String fileUrl) {
            this.fileUrl = fileUrl;
        }

        @Override
        public String toString() {
            return "FileUpload{" +
                    "fileUrl='" + fileUrl + '\'' +
                    '}';
        }
    }
}
