package ru.yandex.canvas.controllers.html5;

import java.util.Arrays;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import javax.validation.Valid;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.Pattern;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.mongodb.client.result.UpdateResult;
import org.springframework.data.domain.Sort;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.canvas.configs.auth.AuthorizeBy;
import ru.yandex.canvas.controllers.OffsetTotalItemsResponse;
import ru.yandex.canvas.exceptions.BadRequestException;
import ru.yandex.canvas.exceptions.NotFoundException;
import ru.yandex.canvas.model.Size;
import ru.yandex.canvas.model.direct.Privileges;
import ru.yandex.canvas.model.html5.Batch;
import ru.yandex.canvas.model.html5.Source;
import ru.yandex.canvas.model.html5.SourceWithId;
import ru.yandex.canvas.model.validation.Html5SizeValidator;
import ru.yandex.canvas.model.validation.NotWhitespace;
import ru.yandex.canvas.service.AuthService;
import ru.yandex.canvas.service.DirectService;
import ru.yandex.canvas.service.SessionParams;
import ru.yandex.canvas.service.html5.Html5BatchesService;
import ru.yandex.canvas.service.html5.Html5SourcesService;

import static ru.yandex.canvas.configs.auth.AuthorizeBy.AuthType.DIRECT_TOKEN;

@RestController
@RequestMapping(path = "/html5")
@Validated
public class Html5BatchesController {
    private static final String SIZE_PATTERN = "(\\d+x\\d+\\s*,\\s*)*\\s*\\d+x\\d+\\s*";

    private final AuthService authService;
    private final Html5BatchesService html5BatchesService;
    private final Html5SourcesService html5SourcesService;
    private final SessionParams sessionParams;
    private final Html5SizeValidator html5SizeValidator;
    private final DirectService directService;

    public Html5BatchesController(Html5BatchesService html5BatchesService, Html5SourcesService html5SourcesService,
                                  AuthService authService, SessionParams sessionParams,
                                  Html5SizeValidator html5SizeValidator, DirectService directService) {
        this.html5BatchesService = html5BatchesService;
        this.html5SourcesService = html5SourcesService;
        this.authService = authService;
        this.sessionParams = sessionParams;
        this.html5SizeValidator = html5SizeValidator;
        this.directService = directService;
    }

    /**
     * @param batchCreateRequest {"name":"Новый креатив","sources":[{"id":"5bd61632fafabf8a5cd10777"}]}
     * @return ResponseEntity<Batch>
     */
    @PostMapping(path = "/batch")
    public ResponseEntity<Batch> createBatch(@Valid @RequestBody BatchCreateRequest batchCreateRequest,
                                             @RequestParam("client_id") Long clientId, Boolean isBrandLift) {
        authService.requirePermission(Privileges.Permission.CREATIVE_CREATE);

        return createBatchWithoutAuth(batchCreateRequest, clientId, isBrandLift);
    }

    /**
     * Ручка создания батча из source-ов с авторизацией по мастер-токену
     * @param batchCreateRequest {"name":"Новый креатив","sources":[{"id":"5bd61632fafabf8a5cd10777"}]}
     * @return ResponseEntity<Batch>
     */
    @AuthorizeBy({DIRECT_TOKEN})
    @PostMapping(path = "/internal-batch")
    public ResponseEntity<Batch> internalCreateBatch(@Valid @RequestBody BatchCreateRequest batchCreateRequest,
                                                     @RequestParam("client_id") Long clientId, Boolean isBrandLift) {
        return createBatchWithoutAuth(batchCreateRequest, clientId, isBrandLift);
    }

    private ResponseEntity<Batch> createBatchWithoutAuth(BatchCreateRequest batchCreateRequest,
                                                         Long clientId, Boolean isBrandLift) {
        List<Source> sources = html5SourcesService.getSources(clientId,
                batchCreateRequest.getSources().stream().map(SourceWithId::getId).toArray(String[]::new));

        if (sources.size() != batchCreateRequest.getSources().size()) {
            throw new NotFoundException(); //TODO
        }

        Batch batch = html5BatchesService.createBatchFromSources(clientId,
                batchCreateRequest.getName(),
                sources,
                sessionParams.getHtml5SessionTag(),
                isBrandLift
        );

        return ResponseEntity.status(HttpStatus.CREATED).body(batch);
    }


    /**
     * @param batchCreateRequest {"name":"Новый креатив","sources":[{"id":"5bd61632fafabf8a5cd10777"}]}
     * @return ResponseEntity<Batch>
     */
    @PostMapping(path = "/batch/survey")
    public ResponseEntity<Batch> createSurveyBatch(@Valid @RequestBody BatchCreateRequest batchCreateRequest,
                                                   @RequestParam("client_id") Long clientId) {
        return createBatch(batchCreateRequest, clientId, true);
    }

    /**
     * curl -v "http://localhost:8080/html5/batch/5a27d41a314d9c004c532697?user_id=221776172&client_id=16948833" | jq .
     */
    @GetMapping(path = "/batch/{id}")
    public ResponseEntity<Batch> getBatch(
            @PathVariable("id") String id,
            @RequestParam("client_id") Long clientId) {
        authService.requirePermission(Privileges.Permission.CREATIVE_GET);
        Batch batch = html5BatchesService.getBatch(id, clientId, sessionParams.getHtml5SessionTag());
        return ResponseEntity.ok(batch);
    }

    /**
     * curl -v -X PATCH -d'{"name": "New creative1"}' "http://localhost:8080/html5/batch/5a27d41a314d9c004c532697
     * ?user_id=221776172&client_id=16948833&name=NewCreativeName3"
     */
    @PatchMapping(path = "/batch/{id}")
    public ResponseEntity renameBatch(
            @PathVariable("id") String id,
            @RequestParam("client_id") Long clientId,
            @RequestBody @Valid BatchRenameRequest batchRenameRequest
    ) {
        authService.requirePermission(Privileges.Permission.CREATIVE_CREATE);

        UpdateResult writeResult
                = html5BatchesService
                .updateBatchName(id, clientId, sessionParams.getHtml5SessionTag(), batchRenameRequest.name);

        return writeResult.getMatchedCount() > 0
                ? ResponseEntity.ok().build()
                : ResponseEntity.notFound().build(); //TODO: maybe error msg? (not used in python)
    }

    /**
     * curl -v -X DELETE "http://localhost:8080/html5/batch/5a27d41a314d9c004c532697?user_id=221776172&client_id
     * =16948833"
     */
    @DeleteMapping(path = "/batch/{id}")
    public ResponseEntity deleteBatch(
            @PathVariable("id") String id,
            @RequestParam("client_id") Long clientId) {
        authService.requirePermission(Privileges.Permission.CREATIVE_CREATE);

        html5BatchesService.archiveBatch(id, clientId, sessionParams.getHtml5SessionTag());

        return ResponseEntity.status(HttpStatus.NO_CONTENT).build();
    }

    /**
     * curl "http://localhost:8080/html5/batches/?user_id=221776172&client_id=16948833&limit=10&offset=0&name
     * =креатив"  | jq .
     * curl -v 'http://localhost:8080/html5/batches?offset=0&limit=50&shared_data=%7B%22clientId%22%3A%223611918%22
     * %2C%22creativeType%22%3A%22common%22%2C%22isCompact%22%3Afalse%2C%22filters%22%3A%7B%22sizes%22%3A%5B%5D%7D%2C
     * %22areFiltersDisabled%22%3Afalse%7D&user_id=221776172&client_id=3611918'
     */
    @GetMapping(path = "/batches")
    public ResponseEntity<OffsetTotalItemsResponse<Batch>> listBatches(
            @RequestParam("client_id") Long clientId,

            @RequestParam(name = "limit", defaultValue = "50") Integer limit,
            @RequestParam(name = "offset", defaultValue = "0") Integer offset,
            @RequestParam(name = "sort_order", defaultValue = "desc") String sortOrder,
            @RequestParam(name = "archive", defaultValue = "false") Boolean archive,
            @RequestParam(name = "sizes", required = false)
            @Valid
            @Pattern(regexp = SIZE_PATTERN) String sizes,
            @RequestParam(name = "name", defaultValue = "") String name
    ) {
        authService.requirePermission(Privileges.Permission.CREATIVE_GET);

        SessionParams.Html5Tag productType = sessionParams.getHtml5SessionTag();
        Set<String> features = directService.getFeatures(clientId, null);

        List<String> sizesList;

        if (sizes == null) {
            sizesList =
                    html5SizeValidator.validSizes(features)
                            .stream()
                            .map(Size::toFilterString)
                            .collect(Collectors.toList());

        } else {
            sizesList = Arrays.asList(sizes.split(",\\s*"));
        }

        if (!isSizeListValid(sizesList, features, productType)) {
            throw new BadRequestException();
        }

        List<Batch> batches = html5BatchesService
                .getBatches(clientId, limit, offset, Sort.Direction.fromString(sortOrder), archive, sizesList, name,
                        productType);

        // Запрос приходится делать каждый раз, т.к. при offset > количества
        // записей монга возвращает последние записи в количестве limit
        long total = html5BatchesService.getBatchesTotalCount(clientId, archive, name);
        int newOffset = batches.size() + offset;

        // TODO: new_offset надо переделать и возвращать только если действительно есть еще записи выбирая из БД n+1
        return ResponseEntity.ok(new OffsetTotalItemsResponse<>(batches,
                newOffset > total ? (int) total : newOffset,
                total));
    }

    private boolean isSizeListValid(List<String> sizes, Set<String> features, SessionParams.Html5Tag productType) {
        if (sizes == null || sizes.isEmpty()) {
            return true;
        }
        List<Size> sizeList = sizes
                .stream()
                .map(e -> e.split("x"))
                .map(e -> Size.of(Integer.parseInt(e[0]), Integer.parseInt(e[1])))
                .collect(Collectors.toList());

        return html5SizeValidator.isSizesValid(sizeList, features, productType);
    }

    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class BatchRenameRequest {
        @JsonProperty("name")
        @Valid
        @NotEmpty
        @NotWhitespace
        @javax.validation.constraints.Size(min = 1, max = 255)
        String name;
    }

}
