package ru.yandex.canvas.controllers.html5;

import java.io.IOException;
import java.net.URISyntaxException;
import java.util.List;

import javax.validation.Valid;
import javax.validation.constraints.Min;
import javax.validation.constraints.NotNull;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.collect.ImmutableList;
import org.hibernate.validator.constraints.NotBlank;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.ResponseEntity;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.canvas.configs.auth.AuthorizeBy;
import ru.yandex.canvas.exceptions.InternalServerError;
import ru.yandex.canvas.exceptions.NotFoundException;
import ru.yandex.canvas.model.html5.Batch;
import ru.yandex.canvas.model.html5.Creative;
import ru.yandex.canvas.model.html5.Source;
import ru.yandex.canvas.repository.html5.BatchesRepository;
import ru.yandex.canvas.service.AuthService;
import ru.yandex.canvas.service.DirectService;
import ru.yandex.canvas.service.ScreenshooterService;
import ru.yandex.canvas.service.SessionParams;
import ru.yandex.canvas.service.html5.Html5CreativesService;
import ru.yandex.canvas.service.html5.Html5SourcesService;
import ru.yandex.canvas.service.html5.Html5Zip;
import ru.yandex.canvas.service.html5.Preview;
import ru.yandex.canvas.service.html5.UrlData;
import ru.yandex.canvas.service.screenshooters.Html5ScreenshooterHelperService;
import ru.yandex.direct.screenshooter.client.model.ScreenShooterScreenshot;

@Validated
@RestController
@RequestMapping(path = "/html5")
public class Html5CreativeController {

    private static final Logger logger = LoggerFactory.getLogger(Html5CreativeController.class);

    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class Request {

        @Valid
        @NotBlank
        @JsonProperty("nonce")
        String nonce;

        @Valid
        @NotNull
        @JsonProperty("data")
        UrlData data;

        public String getNonce() {
            return nonce;
        }

        public UrlData getData() {
            return data;
        }

        public void setNonce(String nonce) {
            this.nonce = nonce;
        }

        public void setData(UrlData data) {
            this.data = data;
        }
    }

    public static class Response {
        @JsonProperty("content")
        String content;

        @JsonProperty("width")
        Integer width;
        @JsonProperty("height")
        Integer height;

        public Response setContent(String content) {
            this.content = content;
            return this;
        }

        public Response setWidth(Integer width) {
            this.width = width;
            return this;
        }

        public Response setHeight(Integer height) {
            this.height = height;
            return this;
        }
    }

    private BatchesRepository batchesRepository;
    private Html5SourcesService sourcesService;
    private Html5CreativesService creativesService;
    private ScreenshooterService screenshooterService;
    private final AuthService authService;
    private final SessionParams sessionParams;
    private final DirectService directService;
    private final Html5ScreenshooterHelperService html5ScreenshooterHelperService;

    public Html5CreativeController(BatchesRepository batchesRepository,
                                   Html5SourcesService sourcesService,
                                   Html5CreativesService creativesService,
                                   ScreenshooterService screenshooterService,
                                   AuthService authService,
                                   SessionParams sessionParams,
                                   DirectService directService,
                                   Html5ScreenshooterHelperService html5ScreenshooterHelperService) {
        this.batchesRepository = batchesRepository;
        this.sourcesService = sourcesService;
        this.creativesService = creativesService;
        this.screenshooterService = screenshooterService;
        this.authService = authService;
        this.sessionParams = sessionParams;
        this.directService = directService;
        this.html5ScreenshooterHelperService = html5ScreenshooterHelperService;
    }

    /**
     * build standalone preview for a given `creative_id`
     * has no auth requirements whatsoever cause moderation looks it up
     * <p>
     * accepts body with json ::
     * <p>
     * {
     * "data": {
     * "click_url": {
     * "clickUrl1": "...",
     * "clickUrl2": "...",  # optional
     * "clickUrl3": "..."   # optional
     * }
     * }
     * }
     */
    @AuthorizeBy({AuthorizeBy.AuthType.TRUSTED_QUERY_STRING})
    @PostMapping(value = "creative/{id}/preview")
    public Response makePreviewOld(@PathVariable("id") @Valid @Min(1) Long id,
                                   @RequestBody @Valid @NotNull Request request) throws IOException,
            URISyntaxException {
        return makePreview(null, id, request);
    }

    @AuthorizeBy({AuthorizeBy.AuthType.TRUSTED_QUERY_STRING})
    @PostMapping(value = "creative/{batch_id}/{id}/preview")
    public Response makePreview(@PathVariable("batch_id") String batchId,
                                @PathVariable("id") @Valid @Min(1) Long id,
                                @RequestBody @Valid @NotNull Request request) throws IOException, URISyntaxException {
        List<Batch> batches = batchId == null ?
                batchesRepository.getBatchesByCreativeIdsIncludeArchived(ImmutableList.of(id)) :
                batchesRepository.getBatchesByBatchIdsIncludeArchived(ImmutableList.of(batchId));

        if (batches.isEmpty()) {
            throw new NotFoundException("No batches found");
        }

        Source source = batches.get(0).getCreatives().stream()
                .filter(e -> e.getId().equals(id))
                .map(Creative::getSource)
                .findFirst()
                .orElse(null);
        if (source ==null) {
            throw new NotFoundException("No source found");
        }

        Html5Zip html5Zip;

        try {
            sourcesService.downloadZipContent(source);
            html5Zip = source.unzipArchiveContent();
        } catch (IOException | InterruptedException e) {
            logger.info("zip download failed: {}", e.getMessage());
            throw new InternalServerError(); //TODO make exception more informative
        }

        Preview.DataParams dataParams;

        if (request.data.getClickUrls().getClickUrl1() == null) { //Как выяснилось из тестов, нам может придти пустой
            // список clickUrl
            dataParams = null;
        } else {
            dataParams = new Preview.DataParams();
            dataParams.addRecord(1, request.data);
        }

        boolean isCloseButtonPresent = !sessionParams.sessionIs(SessionParams.SessionTag.CPM_YNDX_FRONTPAGE);

        Preview preview = new Preview(source.getHtmlReplacements(),
                html5Zip.getFileAsUtf8String(source.getHtmlFilename()),
                dataParams,
                request.nonce,
                isCloseButtonPresent, directService.getFeatures(source.getClientId(), null),
                null);//null ведь превью приходит без sessionParams от модерации. Менять onload там не хотим

        return new Response()
                .setContent(preview.asHtml(source.getWidth(), source.getHeight(), source.getBasePath()))
                .setHeight(source.getHeight())
                .setWidth(source.getWidth());

    }

    @GetMapping(value = "creative/getScreenshot/{id}")
    public ResponseEntity<byte[]> getScreenshot(@PathVariable("id") Long id) {
        return getScreenshotOld(id, authService.getUserId());
    }

    @GetMapping(value = "creative/{id}/getScreenshot")
    public ResponseEntity<byte[]> getScreenshotOld(@PathVariable("id") Long id, @RequestParam("user_id") long userId) {
        /* Проверяем, что у юзера есть доступ к превью на данном клиенте, чтобы нельзя было
         * смотреть скриншоты чужих креативов.
         * TODO: временно! выключаем проверку прав, подробности в тикете DIRECT-117603
         */
        // authService.requirePermissionDifferentClient(clientId, Privileges.Permission.PREVIEW);
        var creative = creativesService.getCreativeWithClientIdOrThrow(id);
        ScreenShooterScreenshot screenshot = html5ScreenshooterHelperService.getScreenshot(creative.getCreative(),
                userId, creative.getClientId());
        return screenshooterService.prepareScreenshotForResponse(screenshot);
    }

    @GetMapping(value = "creative/validate/{id}")
    public ResponseEntity validateСreative(@PathVariable("id") Long id) {
        var creative = creativesService.getCreativeWithClientIdOrThrow(id);
        creativesService.checkForExternalRequestsAndTrowError(creative.getCreative());
        return ResponseEntity.ok().build();
    }
}
