package ru.yandex.canvas.controllers.html5;

import javax.annotation.Nullable;
import javax.servlet.http.HttpServletResponse;

import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;

import ru.yandex.canvas.configs.auth.AuthorizeBy;
import ru.yandex.canvas.model.direct.Privileges;
import ru.yandex.canvas.model.html5.Source;
import ru.yandex.canvas.service.AuthService;
import ru.yandex.canvas.service.SessionParams;
import ru.yandex.canvas.service.html5.Html5SourcesService;

import static ru.yandex.canvas.configs.auth.AuthorizeBy.AuthType.DIRECT_TOKEN;
import static ru.yandex.direct.utils.CommonUtils.nvl;

@RestController
@RequestMapping(path = "/html5")
public class SourceController {

    private final SessionParams sessionParams;
    private final Html5SourcesService html5SourcesService;
    private final AuthService authService;

    public SourceController(Html5SourcesService html5SourcesService, AuthService authService,
                            SessionParams sessionParams) {
        this.html5SourcesService = html5SourcesService;
        this.authService = authService;
        this.sessionParams = sessionParams;
    }

    @PostMapping(path = "source")
    public Source uploadSource(@RequestParam("attachment") MultipartFile file,
                               @RequestParam("client_id") Long clientId,
                               HttpServletResponse response
    ) {
        authService.requirePermission(Privileges.Permission.CREATIVE_CREATE);
        return uploadSourceWithoutAuth(file, clientId, sessionParams.getHtml5SessionTag(), response);
    }

    /**
     * Ручка для загрузки html5-архива с авторизацией по мастер-токену
     */
    @AuthorizeBy({DIRECT_TOKEN})
    @PostMapping(path = "/internal-source")
    public Source internalUploadSource(
            @RequestParam("attachment") MultipartFile file,
            @RequestParam("client_id") Long clientId,
            @RequestParam(value = "html5_tag", required = false) @Nullable SessionParams.Html5Tag html5Tag,
            HttpServletResponse response
    ) {
        return uploadSourceWithoutAuth(file, clientId, nvl(html5Tag, SessionParams.Html5Tag.PLAYABLE), response);
    }

    private Source uploadSourceWithoutAuth(MultipartFile file,
                                           Long clientId,
                                           SessionParams.Html5Tag html5Tag,
                                           HttpServletResponse response
    ) {
        Source source = html5SourcesService.uploadSource(file, clientId, html5Tag);

        response.setStatus(HttpServletResponse.SC_CREATED);

        return source;
    }

    @GetMapping(path = "source/{id}")
    public Source getSource(@PathVariable("id") String id,
                            @RequestParam("client_id") Long clientId) {
        authService.requirePermission(Privileges.Permission.CREATIVE_GET);
        return html5SourcesService.getSource(id, clientId);
    }

    @DeleteMapping(path = "source/{id}")
    public void deleteSource(@PathVariable("id") String id,
                             @RequestParam("client_id") Long clientId,
                             HttpServletResponse response
    ) {
        authService.requirePermission(Privileges.Permission.CREATIVE_CREATE);
        html5SourcesService.deleteSource(id, clientId);
        response.setStatus(HttpServletResponse.SC_NO_CONTENT);
    }


}
