package ru.yandex.canvas.controllers.overlay;

import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.util.Date;

import javax.validation.Valid;
import javax.validation.constraints.Min;
import javax.validation.constraints.NotNull;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.validation.BindingResult;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;

import ru.yandex.canvas.LocalizedBindingResultBuilder;
import ru.yandex.canvas.controllers.video.DateSerializer;
import ru.yandex.canvas.controllers.video.PreviewResponseEntity;
import ru.yandex.canvas.exceptions.InternalServerError;
import ru.yandex.canvas.exceptions.NotFoundException;
import ru.yandex.canvas.exceptions.ValidationErrorsException;
import ru.yandex.canvas.exceptions.VideoAdditionCreationValidationException;
import ru.yandex.canvas.model.direct.Privileges;
import ru.yandex.canvas.model.video.Addition;
import ru.yandex.canvas.model.video.addition.AdditionData;
import ru.yandex.canvas.model.video.addition.AdditionDataBundle;
import ru.yandex.canvas.model.video.addition.AdditionElement;
import ru.yandex.canvas.model.video.addition.Options;
import ru.yandex.canvas.model.video.overlay.OverlayBundle;
import ru.yandex.canvas.model.video.overlay.OverlayCreative;
import ru.yandex.canvas.service.AuthService;
import ru.yandex.canvas.service.TankerKeySet;
import ru.yandex.canvas.service.video.VideoCreativesService;
import ru.yandex.canvas.service.video.VideoPresetsService;
import ru.yandex.canvas.service.video.overlay.OverlayService;
import ru.yandex.canvas.service.video.overlay.OverlayValidationException;
import ru.yandex.canvas.service.video.presets.PresetTag;
import ru.yandex.canvas.service.video.presets.VideoPreset;

import static java.nio.charset.StandardCharsets.UTF_8;
import static java.util.Collections.singletonList;

/**
 * Оверлейные креативы
 * <p>
 * https://wiki.yandex-team.ru/users/elwood/canvas/overlay/
 */
@RestController
@RequestMapping(path = "/overlay")
@Validated
public class OverlayController {
    private static final Logger logger = LoggerFactory.getLogger(OverlayController.class);

    private static final String VALIDATION_ERROR = "Validation error";

    private final TankerKeySet keyset = TankerKeySet.OVERLAY_VALIDATION_MESSAGES;

    private AuthService authService;
    private OverlayService overlayService;
    private VideoCreativesService videoCreativesService;
    private VideoPresetsService videoPresetsService;

    public OverlayController(AuthService authService,
                             OverlayService overlayService,
                             VideoCreativesService videoCreativesService,
                             VideoPresetsService videoPresetsService) {
        this.authService = authService;
        this.overlayService = overlayService;
        this.videoCreativesService = videoCreativesService;
        this.videoPresetsService = videoPresetsService;
    }

    @PostMapping(path = "/files")
    public ResponseEntity<UploadedOverlay> uploadFile(
            @RequestParam("file") final MultipartFile file,
            @RequestParam("client_id") Long clientId) {
        authService.requirePermission(Privileges.Permission.CREATIVE_CREATE);
        try {
            OverlayBundle overlayBundle = overlayService.processZipFile(file, clientId);
            overlayService.insert(overlayBundle);
            UploadedOverlay result = new UploadedOverlay(
                    overlayBundle.getId(), overlayBundle.getDate(), clientId, overlayBundle.getName());
            return ResponseEntity.status(HttpStatus.CREATED).body(result);
        } catch (OverlayValidationException e) {
            throw new ValidationErrorsException(singletonList(e.getLocalizedMessage()));
        }
    }

    @PostMapping(path = "/preview")
    public PreviewResponseEntity getOverlayPreview(@RequestBody PreviewRequest previewRequest,
                                                   @RequestParam("client_id") Long clientId) {
        authService.requirePermission(Privileges.Permission.CREATIVE_GET);
        //
        BindingResult bindingResult = validatePreviewRequest(previewRequest);
        if (bindingResult.hasErrors()) {
            throw new VideoAdditionCreationValidationException("Validation error", bindingResult);
        }
        String overlayId = previewRequest.getData().getElements().get(0).getOptions().getOverlayId();
        OverlayBundle overlayBundle = overlayService.findById(overlayId, clientId);

        if (overlayBundle == null || !overlayBundle.getClientId().equals(clientId)) {
            LocalizedBindingResultBuilder binder =
                    new LocalizedBindingResultBuilder(previewRequest, "previewRequest", keyset);
            binder.rejectValue("data.elements[0].options.overlayId", "overlay_file_not_found");
            throw new VideoAdditionCreationValidationException(VALIDATION_ERROR, binder.build());
        }

        String vast = overlayBundle.getVast();
        return new PreviewResponseEntity(vast);
    }

    @PostMapping(path = "")
    @ResponseBody
    public ResponseEntity<OverlayCreative> createOverlay(@RequestBody OverlayAddition overlayAddition,
                                                         @NotNull @RequestParam("client_id") Long clientId) {
        authService.requirePermission(Privileges.Permission.CREATIVE_CREATE);

        BindingResult bindingResult = validateOverlayAddition(overlayAddition);
        if (bindingResult.hasErrors()) {
            throw new VideoAdditionCreationValidationException("Validation error", bindingResult);
        }

        String overlayId = overlayAddition.getData().getElements().get(0).getOptions().getOverlayId();

        OverlayBundle overlayBundle = overlayService.findById(overlayId, clientId);

        // Дополнительная валидация
        if (overlayBundle == null || !overlayBundle.getClientId().equals(clientId)) {
            LocalizedBindingResultBuilder binder =
                    new LocalizedBindingResultBuilder(overlayAddition, "addition", keyset);
            binder.rejectValue("data.elements[0].options.overlayId", "overlay_file_not_found");
            throw new VideoAdditionCreationValidationException(VALIDATION_ERROR, binder.build());
        }

        String vast = overlayService.getVast(overlayBundle);

        OverlayCreative overlayCreative = new OverlayCreative()
                .setPresetId(overlayAddition.getPresetId())
                .setClientId(clientId)
                .setOverlayId(overlayId)
                .setArchive(false)
                .setName(overlayAddition.getName())
                .setVast(vast);

        overlayService.save(overlayCreative);

        // Отправляем в Direct и RTBHost как video-addition
        Addition addition = overlayCreativeToAddition(overlayCreative);

        long userId = authService.getUserId();

        try {
            videoCreativesService.uploadAdditionsToDirect(singletonList(addition), userId, clientId);
        } catch (RuntimeException e) {
            logger.error("Overlay IntAPI uploading failed", e);
            throw new InternalServerError(e.getMessage());
        }
        try {
            videoCreativesService.uploadToRtbHost(singletonList(addition));
        } catch (RuntimeException e) {
            logger.error("Overlay RtbHost uploading failed", e);
            throw new InternalServerError(e.getMessage());
        }

        return ResponseEntity.ok(overlayCreative);
    }

    @GetMapping(path = "/presets/{id}")
    public String getPreset(@PathVariable("id") @Valid @Min(1) Long id) throws IOException {
        VideoPreset videoPreset = videoPresetsService.getPreset(id);
        if (videoPreset == null || !videoPreset.getTags().contains(PresetTag.OVERLAY)) {
            throw new NotFoundException();
        }
        // Захардкодили, потому как не получилось получить ровно то, что хочет фронтенд,
        // подпиливая конфиг пресета для video-additions без модификаций самого кода video-additions
        return IOUtils.toString(this.getClass().getResourceAsStream("/overlay/preset_for_frontend.json"), UTF_8);
    }

    /**
     * Первичная валидация input. Вручную, чтобы была рабочая привязка к путям некорректных полей.
     */
    private BindingResult validatePreviewRequest(PreviewRequest previewRequest) {
        LocalizedBindingResultBuilder binder =
                new LocalizedBindingResultBuilder(previewRequest, "previewRequest", keyset);
        if (previewRequest.getPresetId() == null) {
            binder.rejectValue("presetId", "javax.validation.constraints.NotNull.message");
            return binder.build();
        }
        if (previewRequest.getData() == null) {
            binder.rejectValue("data", "javax.validation.constraints.NotNull.message");
            return binder.build();
        }
        validateData(binder);
        return binder.build();
    }

    private void validateData(LocalizedBindingResultBuilder binder) {
        binder.validate("data", (dataObject, dataResult) -> {
            OverlayAdditionData data = (OverlayAdditionData) dataObject;
            if (data.getElements() == null || data.getElements().isEmpty()) {
                dataResult.rejectValue("elements", "org.hibernate.validator.constraints.NotBlank.message");
                return;
            }
            dataResult.validate("elements", (elementObject, elementResult) -> {
                OverlayAdditionElement element = (OverlayAdditionElement) elementObject;
                if (element.getAvailable() == null) {
                    elementResult.rejectValue("available", "javax.validation.constraints.NotNull.message");
                }
                if (element.getType() == null || element.getType().isEmpty()) {
                    elementResult.rejectValue("type", "org.hibernate.validator.constraints.NotBlank.message");
                }
                if (element.getOptions() == null) {
                    elementResult.rejectValue("options", "javax.validation.constraints.NotNull.message");
                    return;
                }
                elementResult.validate("options", (optionsObject, optionsResult) -> {
                    OverlayAdditionElementOptions options = (OverlayAdditionElementOptions) optionsObject;
                    if (options.getOverlayId() == null || options.getOverlayId().isEmpty()) {
                        optionsResult.rejectValue("overlayId", "org.hibernate.validator.constraints.NotBlank.message");
                    }
                });
            });
        });
    }

    /**
     * Первичная валидация input. Вручную, чтобы была рабочая привязка к путям некорректных полей.
     */
    private BindingResult validateOverlayAddition(OverlayAddition overlayAddition) {
        LocalizedBindingResultBuilder binder = new LocalizedBindingResultBuilder(overlayAddition, "addition", keyset);
        if (overlayAddition.getPresetId() == null) {
            binder.rejectValue("presetId", "javax.validation.constraints.NotNull.message");
            return binder.build();
        }
        if (overlayAddition.getName() == null || overlayAddition.getName().isEmpty()) {
            binder.rejectValue("name", "org.hibernate.validator.constraints.NotBlank.message");
            return binder.build();
        }
        if (overlayAddition.getData() == null) {
            binder.rejectValue("data", "javax.validation.constraints.NotNull.message");
            return binder.build();
        }
        validateData(binder);
        return binder.build();
    }

    /**
     * Конвертирует оверлейный креатив в модельку Addition, пригодную для отправки в RTBHost и Direct
     * (для переиспользования соответствующего кода)
     */
    private Addition overlayCreativeToAddition(OverlayCreative overlayCreative) {
        AdditionElement additionElement = new AdditionElement()
                .withAvailable(true)
                .withType(AdditionElement.ElementType.ADDITION);
        try {
            // По этому значению код, отправляющий в RTBHost и Direct,
            // будет в том числе отличать оверлейный креатив от обычного
            additionElement.setOptions(new Options().setOverlayId(overlayCreative.getOverlayId()));
        } catch (NoSuchMethodException | IllegalAccessException | InvocationTargetException | InstantiationException e) {
            throw new RuntimeException(e);
        }

        return new Addition()
                .setId(overlayCreative.getId())
                .setPresetId(overlayCreative.getPresetId())
                .setName(overlayCreative.getName())
                .setClientId(overlayCreative.getClientId())
                .setArchive(false)
                .setCreativeId(overlayCreative.getCreativeId())
                .setCreationTime(overlayCreative.getDate())
                .setDate(overlayCreative.getDate())
                .setVast(overlayCreative.getVast())
                .setPreviewUrl(OverlayService.OVERLAY_PREVIEW_URL)
                .setData(new AdditionData()
                        .setBundle(new AdditionDataBundle().setName("empty-bundle"))
                        .setElements(singletonList(additionElement)));
    }

    static class UploadedOverlay {
        @JsonIgnore
        private String overlayBundleId;

        @JsonIgnore
        private Date date;

        @JsonIgnore
        private long clientId;

        @JsonIgnore
        private String name;

        UploadedOverlay(String overlayBundleId, Date date, long clientId, String name) {
            this.overlayBundleId = overlayBundleId;
            this.date = date;
            this.clientId = clientId;
            this.name = name;
        }

        @JsonProperty("id")
        public String getId() {
            return overlayBundleId;
        }

        @JsonProperty("type")
        public String getType() {
            return "overlay";
        }

        @JsonProperty("date")
        @JsonSerialize(using = DateSerializer.class)
        public Date getDate() {
            return date;
        }

        @JsonProperty("client_id")
        public Long clientId() {
            return clientId;
        }

        @JsonProperty("name")
        public String getName() {
            return name;
        }
    }
}
