package ru.yandex.canvas.controllers.video;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonUnwrapped;

import ru.yandex.canvas.model.video.addition.options.OptionName;
import ru.yandex.canvas.service.VideoLimitsInterface;
import ru.yandex.canvas.service.video.presets.PresetTag;
import ru.yandex.canvas.service.video.presets.VideoPreset;
import ru.yandex.canvas.service.video.presets.configs.BaseConfig;
import ru.yandex.canvas.service.video.presets.configs.ConfigType;
import ru.yandex.canvas.service.video.presets.configs.options.ColorPickerOptionConfig;
import ru.yandex.canvas.service.video.presets.configs.options.CustomLabelOptionConfig;
import ru.yandex.canvas.service.video.presets.configs.options.OptionConfig;
import ru.yandex.canvas.service.video.presets.configs.options.PositionOptionConfig;
import ru.yandex.canvas.service.video.presets.elements.BaseElementOptions;
import ru.yandex.canvas.service.video.presets.elements.ElementTextOptions;
import ru.yandex.canvas.service.video.presets.elements.PresetElement;
import ru.yandex.canvas.service.video.presets.elements.PresetElementType;
import ru.yandex.direct.feature.FeatureName;

public class PresetWrapper {
    @JsonIgnore
    private VideoPreset videoPreset;

    @JsonIgnore
    private VideoLimitsInterface limits;

    @JsonIgnore
    private Function<String, String> translator;

    @JsonIgnore
    private Set<String> features;

    public PresetWrapper(VideoPreset preset, Function<String, String> translator, VideoLimitsInterface limits,
                         Set<String> features) {
        this.videoPreset = preset;
        this.translator = translator;
        this.limits = limits;
        this.features = features;
    }

    @JsonProperty("id")
    public String getId() {
        return videoPreset.getId().toString();
    }

    @JsonProperty("thumbnail")
    public String getThumbnail() {
        return translator.apply(videoPreset.getThumbnail());
    }

    @JsonProperty("tags")
    public List<PresetTag> getTags() {
        return new ArrayList<>(videoPreset.getTags());
    }

    @JsonProperty("skip_offset")
    public Long getSkipOffset() {
        return (long) videoPreset.getSkipOffset().asSeconds();
    }

    @JsonProperty("data")
    public Data getData() {
        return new Data(videoPreset.getPresetElementList(), videoPreset.getBundleName(), translator, features);
    }

    @JsonProperty("name")
    public String name() {
        return videoPreset.getName();
    }

    @JsonProperty("allow_stock_video")
    public Boolean allowStockVideo() {
        return videoPreset.getAllowStockVideo();
    }

    @JsonProperty("geometry")
    public String getGeometry() {
        return videoPreset.getDescription().getGeometry().getValue().toLowerCase();
    }

    @JsonProperty("duration_min_sec")
    public Double getDurationMin() {
        return limits.getDurationMin();
    }

    @JsonProperty("duration_max_sec")
    public Double getDurationMax() {
        return limits.getDurationMax();
    }

    @JsonProperty("packshot_present")
    public Boolean isPackshotPresent() {
        return videoPreset.isPackshotAllowed();
    }

    public static class Data {


        public static class Bundle {
            @JsonProperty("name")
            String name;

            public Bundle(String name) {
                this.name = name;
            }
        }

        public static class PresetElementOptionsWrapper {
            @JsonIgnoreProperties({"placeholder"})
            @JsonUnwrapped()
            BaseElementOptions options;

            @JsonIgnore
            Function<String, String> translator;

            @JsonProperty("placeholder")
            @JsonInclude(JsonInclude.Include.NON_NULL)
            String getPlaceholder() {
                //TODO instanceof always looks bad
                if (options instanceof ElementTextOptions) {
                    return translator.apply(((ElementTextOptions) options).getPlaceholder());
                }

                return null;
            }

            public PresetElementOptionsWrapper(BaseElementOptions options, Function<String, String> translator) {
                this.options = options;
                this.translator = translator;
            }

        }

        public static class PresetElementWrapper {
            @JsonIgnoreProperties({"options"})
            @JsonUnwrapped
            PresetElement element;

            @JsonIgnore
            Function<String, String> translator;

            @JsonProperty("options")
            Data.PresetElementOptionsWrapper getOptions() {
                return new Data.PresetElementOptionsWrapper(element.getOptions(), translator);
            }

            public PresetElementWrapper(PresetElement element, Function<String, String> translator) {
                this.element = element;
                this.translator = translator;
            }

        }

        @JsonProperty("elements")
        List<Data.PresetElementWrapper> elements;

        @JsonProperty("bundle")
        Data.Bundle bundle;

        public Data(List<PresetElement> elements, String name, Function<String, String> translator,
                    Set<String> features) {
            this.elements =
                    elements.stream()
                            .filter(it -> it.getType() != PresetElementType.AGE
                                    || !features.contains(FeatureName.HIDE_AGE_LABEL.getName()))
                            .map(e -> new Data.PresetElementWrapper(e, translator)).collect(Collectors.toList());
            this.bundle = new Data.Bundle(name);
        }
    }

    public static class ConfigOptionWrapper {
        @JsonIgnore
        protected Function<String, String> translator;
        protected OptionConfig option;

        @JsonIgnoreProperties({"name", "title", "tooltip", "optionName"})
        @JsonUnwrapped()
        public OptionConfig getOption() {
            return option;
        }

        @JsonProperty("name")
        String getName() {
            return option.getName();
        }

        @JsonProperty("title")
        String getTitle() {
            return translator.apply(option.getTitle());
        }

        @JsonProperty("tooltip")
        String getTooltip() {
            return translator.apply(option.getTooltip());
        }

        ConfigOptionWrapper(OptionConfig option, Function<String, String> translator) {
            this.option = option;
            this.translator = translator;
        }
    }


    public static class CustomLabelConfigOptionWrapper extends ConfigOptionWrapper {

        @JsonIgnoreProperties({"name", "title", "tooltip", "optionName", "defaultValue", "allowedValues"})
        @JsonUnwrapped()
        public OptionConfig getOption() {
            return option;
        }

        @JsonProperty("defaultValue")
        String getDefaultValue() {
            return translator.apply(option.getDefaultValue());
        }

        @JsonProperty("allowedValues")
        List<String> getAllowedValues() {
            List<String> translated = new ArrayList<>();
            for (String value : option.getAllowedValues()) {
                translated.add(translator.apply(value));
            }
            return translated;
        }

        CustomLabelConfigOptionWrapper(OptionConfig option, Function<String, String> translator) {
            super(option, translator);
        }
    }

    public static class ConfigWrapper {

        ConfigWrapper(BaseConfig config, Function<String, String> translator, Set<String> features) {
            this.config = config;
            this.translator = translator;
            this.features = features;
        }

        @JsonIgnore
        Function<String, String> translator;

        @JsonIgnoreProperties({"options", "title", "tooltip"})
        @JsonUnwrapped()
        BaseConfig config;

        @JsonIgnore
        private Set<String> features;

        ConfigOptionWrapper configToWrapper(OptionConfig e) {
            if (e instanceof CustomLabelOptionConfig) {
                return new CustomLabelConfigOptionWrapper(e, translator);
            }
            if (config.getConfigType() == ConfigType.BUTTON) {
                if (e.getOptionName() == OptionName.COLOR || e.getOptionName() == OptionName.TEXT_COLOR) {
                    e = ColorPickerOptionConfig.builder()
                            .setTitle(e.getTitle())
                            .setName(e.getName())
                            .setRequired(true)
                            .setTooltip(e.getTooltip())
                            .setEditable(false)
                            .setVisible(false)
                            .setAllowedValues(e.getAllowedValues())
                            .setDefaultValue(e.getDefaultValue())
                            .build();
                }
                if (e.getOptionName() == OptionName.POSITION) {
                    e = PositionOptionConfig.builder()
                            .setTitle(e.getTitle())
                            .setName(e.getName())
                            .setEditable(false)
                            .setVisible(false)
                            .setAllowedValues(e.getAllowedValues())
                            .setDefaultValue(e.getDefaultValue())
                            .build();
                }
            }
            var wrapper =  new ConfigOptionWrapper(e, translator);
            return wrapper;
        }

        @JsonProperty("options")
        List<ConfigOptionWrapper> getOptions() {

            return config.getOptionConfigs().stream()
                    //Placeholders options shouldn't be exposed (they mean nothing to frontend).
                    .filter(e -> e.getOptionName() != OptionName.PLACEHOLDER)
                    .map(this::configToWrapper)
                    .collect(Collectors.toList());
        }

        @JsonProperty("title")
        String getTitle() {
            return translator.apply(config.getTitle());
        }

        @JsonProperty("tooltip")
        String getTooltip() {
            return translator.apply(config.getTooltip());
        }

    }

    @JsonProperty("config")
    public Map<String, ConfigWrapper> getConfig() {
        return videoPreset.getConfig().getConfigs().values().stream()
                .filter(it -> it.getConfigType() != ConfigType.AGE
                        || !features.contains(FeatureName.HIDE_AGE_LABEL.getName()))
                .collect(Collectors.toMap(e -> e.getConfigType().toString().toLowerCase(),
                        e -> new ConfigWrapper(e, translator, features)));
    }

}
