package ru.yandex.canvas.controllers.video;

import java.io.IOException;
import java.io.Reader;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.Enumeration;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.client.HttpClientErrorException;
import org.springframework.web.client.RestTemplate;

import ru.yandex.canvas.exceptions.InternalServerError;
import ru.yandex.canvas.service.AuthRequestParams;

public class PythonRedirect implements InitializingBean {
    private static PythonRedirect beanHolder = null;

    private static final Logger logger = LoggerFactory.getLogger(PythonRedirect.class);

    private String pythonHost;
    private int pythonPort;
    private RestTemplate restTemplate;
    private AuthRequestParams authRequestParams;

    public PythonRedirect(String pythonBackend, RestTemplate restTemplate) throws MalformedURLException {

        URL url = new URL(pythonBackend);

        pythonHost = url.getHost(); // "127.0.0.32";
        pythonPort = url.getPort(); // 11362
        this.restTemplate = restTemplate;
    }

    private String getFullURL(HttpServletRequest request) {
        StringBuilder requestURL = new StringBuilder(request.getRequestURL().toString());
        String queryString = request.getQueryString();

        if (queryString == null) {
            return requestURL.toString();
        } else {
            return requestURL.append('?').append(queryString).toString();
        }
    }

    public static ResponseEntity route(HttpServletRequest request) throws IOException {
        return beanHolder.redirectRequest(request);
    }

    public ResponseEntity redirectRequest(HttpServletRequest request) throws IOException {
        try (Reader reader = request.getReader()) {

            String body = IOUtils.toString(reader);
            URI oldUri = new URI(getFullURL(request));

            String path = oldUri.getPath().replaceAll("^/video", "");

            URI newUri = new URI(oldUri.getScheme(), oldUri.getUserInfo(), pythonHost, pythonPort,
                    path, oldUri.getQuery(), oldUri.getFragment());

            String url = newUri.toString();

            logger.info("Unmapped request handling: {} {} {}", url, path, body);

            MultiValueMap<String, String> headers = new LinkedMultiValueMap<>();
            Enumeration<String> headerNames = request.getHeaderNames();

            if (headerNames != null) {
                while (headerNames.hasMoreElements()) {
                    String headerName = headerNames.nextElement();

                    if (headerName.equals(HttpHeaders.CONTENT_TYPE) && request.getHeader(headerName).contains("json")) {
                        headers.add(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON_UTF8_VALUE);
                    } else {
                        headers.add(headerName, request.getHeader(headerName));
                    }
                }
            }

            ResponseEntity<String> exchange = restTemplate.exchange(url,
                    HttpMethod.valueOf(request.getMethod()),
                    new HttpEntity<>(body, headers),
                    String.class,
                    request.getParameterMap());

            return exchange;
        } catch (final URISyntaxException e) {
            throw new InternalServerError("PythonProxy: " + e.getMessage());
        } catch (final HttpClientErrorException e) {
            return new ResponseEntity<>(e.getResponseBodyAsByteArray(), e.getResponseHeaders(), e.getStatusCode());
        }
    }

    @Override
    public void afterPropertiesSet() {
        beanHolder = this;
    }
}
