package ru.yandex.canvas.controllers.video;

import java.io.IOException;
import java.net.URISyntaxException;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.validation.BindingResult;

import ru.yandex.canvas.exceptions.InternalServerError;
import ru.yandex.canvas.exceptions.VideoAdditionCreationValidationException;
import ru.yandex.canvas.model.video.Addition;
import ru.yandex.canvas.model.video.CustomVastParams;
import ru.yandex.canvas.model.video.addition.AdditionElement;
import ru.yandex.canvas.model.video.files.Movie;
import ru.yandex.canvas.service.video.VideoAdditionValidationService;
import ru.yandex.canvas.service.video.VideoAdditionsService;
import ru.yandex.canvas.service.video.VideoCreativesService;
import ru.yandex.canvas.service.video.VideoPresetsService;
import ru.yandex.canvas.service.video.presets.VideoPreset;

@Service
public class VideoAdditionCreatorService {

    private static final Logger logger = LoggerFactory.getLogger(VideoAdditionCreatorService.class);

    private final VideoAdditionsService videoAdditionsService;
    private final VideoAdditionValidationService videoAdditionValidationService;
    private final VideoCreativesService videoCreativesService;
    private final VideoPresetsService presetsService;

    private static final String ADDITION_VALIDATION_ERROR_MSG = "Validation error";

    public VideoAdditionCreatorService(VideoAdditionsService videoAdditionsService,
                                       VideoAdditionValidationService videoAdditionValidationService,
                                       VideoCreativesService videoCreativesService,
                                       VideoPresetsService presetsService) {
        this.videoAdditionsService = videoAdditionsService;
        this.videoAdditionValidationService = videoAdditionValidationService;
        this.videoCreativesService = videoCreativesService;
        this.presetsService = presetsService;
    }

    public void createAddition(Long userId, Long clientId, Addition addition, CustomVastParams customVastParams) {
        addition.trimTextOptions();

        addition.setClientId(clientId);

        checkAddition(addition, null);

        try {
            videoAdditionsService.createAdditionWithScreenshot(clientId, addition, customVastParams);
        } catch (IOException | URISyntaxException e) {
            logger.error("Addition creation failed", e);
            throw new InternalServerError(e.getMessage());
        }

        // В питоне не валидируются лишние (чье описание отсутствует в пресете) элементы дополнения, повторояем это
        // поведение, но удаляем их, чтобы не записать в БД.
        List<AdditionElement.ElementType> allowedTypes =
                videoAdditionValidationService.getAllowedElements(addition);
        addition.getData().setElements(
                addition.getData().getElements().stream().filter(e -> allowedTypes.contains(e.getType()))
                        .collect(Collectors.toList()));

        videoAdditionsService.saveAddition(addition);

        try {
            videoCreativesService.uploadAdditionsToDirect(Collections.singletonList(addition), userId, clientId);
        } catch (Exception e) {
            logger.error("Addition IntAPI uploading failed ", e);
            throw new InternalServerError(e.getMessage());
        }
        try {
            boolean uploadToRtb = true;
            VideoPreset preset = presetsService.getPreset(addition.getPresetId());
            if (!preset.getDescription().getCpmAudio()) {
                Movie movie = videoAdditionsService.getMovie(addition);
                uploadToRtb = movie == null || movie.isReady();
            }
            // если видео еще не сконвертировано, то не отправляем в rtb
            if (uploadToRtb) {
                videoCreativesService.uploadToRtbHost(Collections.singletonList(addition));
            }
        } catch (Exception e) {
            logger.error("Addition RtbHost uploading failed ", e);
            throw new InternalServerError(e.getMessage());
        }
        videoAdditionsService.afterAdditionCreate(addition);
    }

    public void checkAddition(Addition addition, VideoAdditionValidationService.ValidationOptions options) {
        BindingResult validationResult = videoAdditionValidationService
                .validate(new VideoAdditionValidationService.AdditionValidationObject(addition, options));

        if (validationResult.hasErrors()) {
            throw new VideoAdditionCreationValidationException(ADDITION_VALIDATION_ERROR_MSG, validationResult);
        }
    }
}
