package ru.yandex.canvas.controllers.video;

import java.io.IOException;
import java.net.URISyntaxException;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.servlet.http.HttpServletRequest;
import javax.validation.Valid;
import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonUnwrapped;
import com.google.common.collect.ImmutableMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.domain.Sort;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.validation.BindingResult;
import org.springframework.web.HttpMediaTypeNotAcceptableException;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.canvas.configs.auth.AuthorizeBy;
import ru.yandex.canvas.controllers.OffsetTotalItemsResponse;
import ru.yandex.canvas.exceptions.InternalServerError;
import ru.yandex.canvas.exceptions.NotFoundException;
import ru.yandex.canvas.exceptions.VideoAdditionCreationValidationException;
import ru.yandex.canvas.model.direct.CreativeCampaignResult;
import ru.yandex.canvas.model.direct.Privileges;
import ru.yandex.canvas.model.stillage.StillageFileInfo;
import ru.yandex.canvas.model.validation.ValidUnicodeSymbols;
import ru.yandex.canvas.model.video.Addition;
import ru.yandex.canvas.model.video.CustomVastParams;
import ru.yandex.canvas.model.video.addition.AdditionData;
import ru.yandex.canvas.model.video.addition.AdditionElement;
import ru.yandex.canvas.model.video.addition.options.ButtonElementOptions;
import ru.yandex.canvas.repository.ItemsWithTotal;
import ru.yandex.canvas.service.AuthService;
import ru.yandex.canvas.service.DirectService;
import ru.yandex.canvas.service.SessionParams;
import ru.yandex.canvas.service.StillageService;
import ru.yandex.canvas.service.video.PreviewData;
import ru.yandex.canvas.service.video.StaticPreviewService;
import ru.yandex.canvas.service.video.VideoAdditionsService;
import ru.yandex.canvas.service.video.VideoCreativeType;
import ru.yandex.canvas.service.video.VideoCreativesService;
import ru.yandex.canvas.service.video.VideoPresetsService;
import ru.yandex.canvas.service.video.VideoPreviewUrlBuilder;
import ru.yandex.canvas.service.video.presets.PresetTag;
import ru.yandex.canvas.service.video.presets.VideoPreset;
import ru.yandex.canvas.service.video.presets.configs.BaseConfig;
import ru.yandex.canvas.service.video.presets.configs.ConfigType;
import ru.yandex.canvas.service.video.presets.configs.options.CustomLabelOptionConfig;

import static ru.yandex.canvas.VideoConstants.DEFAULT_VPAID_PCODE_URL;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@RestController
@RequestMapping(path = "/video/additions")
public class VideoAdditionsController {
    private static final Logger logger = LoggerFactory.getLogger(VideoAdditionsController.class);
    private final AuthService authService;
    private final VideoAdditionsService videoAdditionsService;
    private final StillageService stillageService;
    private final VideoCreativesService videoCreativesService;
    private final VideoPresetsService videoPresetsService;
    private final VideoAdditionCreatorService videoAdditionCreatorService;
    private final SessionParams sessionParams;
    private final StaticPreviewService staticPreviewService;
    private final VideoPreviewUrlBuilder videoPreviewUrlBuilder;
    private final DirectService directService;

    private static final String ADDITION_VALIDATION_ERROR_MSG = "Validation error";

    public VideoAdditionsController(AuthService authService,
                                    VideoAdditionsService videoAdditionsService,
                                    StillageService stillageService,
                                    VideoCreativesService videoCreativesService,
                                    VideoAdditionCreatorService videoAdditionCreatorService,
                                    VideoPresetsService videoPresetsService, SessionParams sessionParams,
                                    StaticPreviewService staticPreviewService,
                                    VideoPreviewUrlBuilder videoPreviewUrlBuilder,
                                    DirectService directService) {
        this.authService = authService;
        this.videoAdditionsService = videoAdditionsService;
        this.stillageService = stillageService;
        this.videoCreativesService = videoCreativesService;
        this.videoPresetsService = videoPresetsService;
        this.videoAdditionCreatorService = videoAdditionCreatorService;
        this.sessionParams = sessionParams;
        this.staticPreviewService = staticPreviewService;
        this.videoPreviewUrlBuilder = videoPreviewUrlBuilder;
        this.directService = directService;
    }

    /**
     * `sort_order`: either `asc` or `desc` to sort by date
     *
     * @return
     */
    @GetMapping(path = "")
    public ResponseEntity<OffsetTotalItemsResponse<VideoAdditionItemResponse>> listAdditions(
            @RequestParam("client_id") Long clientId,
            @RequestParam(name = "limit", defaultValue = "50") Integer limit,
            @RequestParam(name = "offset", defaultValue = "0") Integer offset,
            @RequestParam(name = "sort_order", defaultValue = "desc") String sortOrder,
            @RequestParam(name = "archive", defaultValue = "false") Boolean archive,
            @RequestParam(name = "scroll_to", required = false) Integer scrollTo,
            // it's array of filters but with only one possible value
            @RequestParam(name = "filters", required = false) List<String> filters,
            @RequestParam(name = "part", defaultValue = "") String namePart,
            HttpServletRequest httpRequest
    ) {
        authService.requirePermission(Privileges.Permission.CREATIVE_GET);

        if (offset > 0) {
            offset--;
        }

        ItemsWithTotal<Addition> additionsWithTotal = videoAdditionsService
                .getAdditions(authService.getUserId(), clientId, limit, offset, Sort.Direction.fromString(sortOrder),
                        archive, namePart,
                        true,
                        getProperPresetsList(sessionParams), scrollTo,
                        filters != null && filters.contains("campaign_tied"));

        // TODO: new_offset надо переделать и возвращать только если действительно есть еще записи выбирая из БД n+1
        // TODO: посмотреть на
        //  265                 newOffsetLessThanTotal = newOffset < total;
        // кажется тут полная лажа - считать весь total - может быть дорого, а без него newOffset == total
        // лучше поправить и тут и на фронте, пример с большим количеством креативово: clientid: 493008

        int newOffset = (int) Math.min(additionsWithTotal.getRealOffset() + 1, additionsWithTotal.getTotal());

        ResponseEntity<OffsetTotalItemsResponse<VideoAdditionItemResponse>> response = ResponseEntity.ok(
                new OffsetTotalItemsResponse<>(

                        additionsWithTotal.getItems().stream()
                                .map(e -> {
                                    VideoAdditionItemResponse resp = new VideoAdditionItemResponse(e);
                                    resp.setPreviewUrl(videoPreviewUrlBuilder.getPreviewUrl(clientId, e, false));
                                    var preset = videoPresetsService.getPreset(e.getPresetId());
                                    if (preset.getDescription().getCpmAudio()) {
                                        resp.setHasPackshot(false);
                                    }
                                    return resp;
                                })
                                .collect(Collectors.toList()),
                        newOffset,
                        additionsWithTotal.getTotal()));

        return response;
    }

    public static class GetAdditionResponse {
        @JsonUnwrapped
        private VideoAdditionItemResponse videoAdditionItemResponse;

        @JsonIgnore
        private VideoPreset videoPreset;

        @JsonIgnore
        private VideoPresetsService videoPresetsService;

        @JsonIgnore
        private Set<String> features;

        @JsonProperty("config")
        public Map<String, PresetWrapper.ConfigWrapper> getConfig() {
            VideoCreativeType creativeType = videoPresetsService.fromPresetId(videoPreset.getId());
            return videoPreset.getConfig().getConfigs().values().stream()
                    .collect(Collectors.toMap(e -> e.getConfigType().toString().toLowerCase(),
                            e -> new PresetWrapper.ConfigWrapper(e,
                                    tooltip -> videoPresetsService.tooltipInterpolator(tooltip, creativeType, videoPreset.getId()),
                                    features)));
        }

        public GetAdditionResponse(VideoAdditionItemResponse videoAdditionItemResponse, VideoPreset videoPreset,
                                   VideoPresetsService videoPresetsService, Set<String> features) {
            this.videoAdditionItemResponse = videoAdditionItemResponse;
            this.videoPreset = videoPreset;
            this.videoPresetsService = videoPresetsService;
            this.features = features;
        }

        public VideoAdditionItemResponse getVideoAdditionItemResponse() {
            return videoAdditionItemResponse;
        }

        public GetAdditionResponse setVideoAdditionItemResponse(VideoAdditionItemResponse videoAdditionItemResponse) {
            this.videoAdditionItemResponse = videoAdditionItemResponse;
            return this;
        }
    }

    private List<Long> getProperPresetsList(SessionParams sessionParams) {
        if (sessionParams.getVideoPresetIds() != null) {
            return sessionParams.getVideoPresetIds().stream()
                    .peek(e ->  { if (!videoPresetsService.contains(e)) {throw new NotFoundException();} } )
                    .collect(Collectors.toList());
        }

        if (sessionParams.sessionIs(SessionParams.SessionTag.CPM_AUDIO)) {
            return videoPresetsService.getPresetsByTag(PresetTag.CPM_AUDIO).stream().map(VideoPreset::getId).collect(
                    Collectors.toList());
        }

        return videoPresetsService.getPresetsByCreativeType(sessionParams.getCreativeType()).stream().map(VideoPreset::getId).collect(
                Collectors.toList());
    }

    @GetMapping(path = "/{id}")
    public ResponseEntity<GetAdditionResponse> getAddition(@PathVariable(value = "id") @Valid @NotBlank String id,
                                                           @RequestParam("client_id") Long clientId
    ) {
        authService.requirePermission(Privileges.Permission.CREATIVE_CREATE);
        Set<String> features = directService.getFeatures(sessionParams.getClientId(), null);
        Addition addition = videoAdditionsService.getAddition(clientId, id);

        if (addition == null) {
            throw new NotFoundException("Specified addition (id=" + id + ", client_id=" + clientId + ") not found");
        }

        fitAdditionToPreset(addition);

        return ResponseEntity.ok(new GetAdditionResponse(new VideoAdditionItemResponse(addition),
                videoPresetsService.getPreset(addition.getPresetId()), videoPresetsService, features));
    }

    private void fitAdditionToPreset(Addition addition) {
        AdditionData data = addition.getData();
        if (data == null) {
            return;
        }
        List<AdditionElement> elements = data.getElements();
        if (elements == null) {
            return;
        }

        VideoCreativeType creativeType = videoPresetsService.fromPresetId(addition.getPresetId());

        for (AdditionElement element : elements) {
            if (AdditionElement.ElementType.BUTTON.equals(element.getType())) {
                ButtonElementOptions options = (ButtonElementOptions) element.getOptions();
                String customLabel = options.getCustomLabel();
                VideoPreset preset = videoPresetsService.getPreset(addition.getPresetId());
                BaseConfig buttonConfig = preset.getConfig().getConfigs().get(ConfigType.BUTTON);
                if (buttonConfig == null) {
                    return;
                }
                CustomLabelOptionConfig config = (CustomLabelOptionConfig) buttonConfig.getOptionConfigs().stream()
                        .filter(e -> e instanceof CustomLabelOptionConfig).findAny().orElse(null);
                if (config == null) {
                    return;
                }
                List<String> localizedValues = mapList(config.getAllowedValues(),
                        e -> videoPresetsService.tooltipInterpolator(e, creativeType, addition.getPresetId()));
                if (localizedValues == null) {
                    return;
                }

                if (!localizedValues.contains(customLabel)) {
                    options.setCustomLabel(videoPresetsService.tooltipInterpolator(config.getDefaultValue(),
                            creativeType, addition.getPresetId()));
                }
            }
        }
    }

    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class RenameRequest {
        @ValidUnicodeSymbols
        @NotEmpty
        @JsonProperty("name")
        private String name;

        public String getName() {
            return name;
        }

        public void setName(String name) {
            this.name = name;
        }
    }

    @PatchMapping(path = "/{id}")
    public ResponseEntity<String> changeAdditionName(@PathVariable(value = "id") String id,
                                                     @RequestBody @Valid RenameRequest renameRequest,
                                                     @RequestParam("client_id") Long clientId
    ) {
        authService.requirePermission(Privileges.Permission.CREATIVE_CREATE);

        if (!videoAdditionsService.renameAddition(clientId, id, renameRequest.getName())) {
            //app.logger.warn("Specified addition (id=%s, client_id=%s) not found" % (addition_id, client))
            throw new NotFoundException("Specified addition (id=" + id + ", client_id=" + clientId + ") not found");
        }

        videoCreativesService.uploadToDirect(Collections.singletonList(id), authService.getUserId(), clientId);

        return ResponseEntity.ok("");
    }

    /**
     * curl -H "Content-Type: application/json" -X POST -d @test_addition.json
     * "http://localhost:8080/video/additions?user_id=221776172&client_id=3641429"
     *
     * @param addition
     * @return
     */
    @PostMapping(path = "")
    public ResponseEntity<Addition> createAddition(
            @RequestParam("client_id") Long clientId,
            @RequestBody @Valid Addition addition,
            final BindingResult bindingResult
    ) {
        authService.requirePermission(Privileges.Permission.CREATIVE_CREATE);

        if (bindingResult.hasErrors()) {
            throw new VideoAdditionCreationValidationException(ADDITION_VALIDATION_ERROR_MSG, bindingResult);
        }

        videoAdditionCreatorService.createAddition(authService.getUserId(), clientId, addition, new CustomVastParams()
                .setVpaidPcodeUrl(DEFAULT_VPAID_PCODE_URL));

        return ResponseEntity.ok(addition);
    }


    public static class PreviewRequest {
        @NotNull
        @Valid
        @JsonProperty("data")
        private AdditionData data;

        @NotNull
        @Valid
        @JsonProperty("previewData")
        private PreviewData previewData;

        @NotNull
        @JsonProperty("preset_id")
        private Long presetId;

        public Long getPresetId() {
            return presetId;
        }

        public PreviewRequest setPresetId(Long presetId) {
            this.presetId = presetId;
            return this;
        }

        public AdditionData getData() {
            return data;
        }

        public PreviewRequest setData(AdditionData data) {
            this.data = data;
            return this;
        }

        public PreviewData getPreviewData() {
            return previewData;
        }

        public PreviewRequest setPreviewData(PreviewData previewData) {
            this.previewData = previewData;
            return this;
        }
    }

    @PostMapping(path = "/preview")
    public PreviewResponseEntity getAdditionPreview(@Valid @RequestBody PreviewRequest previewRequest,
                                                    @RequestParam("client_id") Long clientId
    ) throws IOException {
        authService.requirePermission(Privileges.Permission.CREATIVE_GET);

        Addition addition = new Addition();
        addition.setData(previewRequest.data).setPresetId(previewRequest.getPresetId());

        addition.setClientId(clientId);

        videoAdditionsService.createAddition(clientId, addition, 1L, new CustomVastParams()
                .setVpaidPcodeUrl(DEFAULT_VPAID_PCODE_URL));

        String vast = videoAdditionsService.getPreview(addition, previewRequest.getPreviewData());

        return new PreviewResponseEntity(vast, videoAdditionsService.getRatio(addition));
    }

    @AuthorizeBy({AuthorizeBy.AuthType.TRUSTED_QUERY_STRING})
    @PostMapping(path = "/{id}/preview")
    public PreviewResponseEntity getAdditionPreview(@PathVariable(value = "id") String id,
                                                    @Valid @RequestBody PreviewAddtionRequestEntity previewRequest
    ) throws IOException {
        if (previewRequest.getPreviewData() == null) {
            previewRequest.setPreviewData(new PreviewData());
        }

        Addition addition = videoAdditionsService.getAdditionByIdArchivedAlso(id);
        if (addition == null) {
            throw new NotFoundException();
        }

        String vast = videoAdditionsService.getPreview(addition, previewRequest.getPreviewData());

        if (vast == null) {
            throw new NotFoundException();
        }
        return new PreviewResponseEntity(vast, videoAdditionsService.getRatio(addition));
    }

    public static class StaticPreviewEntity {
        @JsonProperty("url")
        private String url;

        public StaticPreviewEntity(String url) {
            this.url = url;
        }
    }

    @PostMapping(path = "/{id}/staticPreview")
    public ResponseEntity<StaticPreviewEntity> getStaticPreview(@PathVariable(value = "id") String id,
                                                                @Valid @RequestBody PreviewRequestEntity previewRequest) {
        Addition addition = videoAdditionsService.getAddition(id);

        if (addition == null) {
            throw new NotFoundException();
        }

        try {
            String result = staticPreviewService.getStaticPreview(addition, previewRequest.getPreviewData());
            StillageFileInfo stillageFileInfo = stillageService.uploadFile("preview.html", result.getBytes());
            return ResponseEntity.ok(new StaticPreviewEntity(stillageFileInfo.getUrl()));
        } catch (IOException | URISyntaxException e) {
            logger.warn("getStaticPreview", e);
            throw new InternalServerError(e.getMessage());
        }
    }

    @AuthorizeBy({AuthorizeBy.AuthType.DIRECT_TOKEN})
    @PatchMapping(path = "/{id}/vast")
    public ResponseEntity<?> patchVast(@PathVariable(value = "id") String id,
                                       @RequestParam(value = "type", required = false) String type) {

        Addition addition;
        try {
            if ("stock".equals(type)) {
                addition = videoCreativesService.patchStockAdditionVastById(id);
            } else {
                addition = videoCreativesService.patchAdditionVastById(id);
            }
        } catch (HttpMediaTypeNotAcceptableException e) {
            logger.warn("patchVast", e);
            return ResponseEntity.status(HttpStatus.NOT_ACCEPTABLE).body(null);
        } catch (IOException | URISyntaxException e) {
            logger.warn("patchVast", e);
            throw new InternalServerError(e.getMessage());
        }

        videoCreativesService.uploadToRtbHost(Collections.singletonList(addition));

        String collection = "stock".equals(type) ? "stock_video_additions" : "video_additions";
        String response = "Vast updated for " + collection + " (id=" + id + ")";

        return ResponseEntity.ok(ImmutableMap.of("message", response));
    }

    public static class CampaignsListEntity {
        List<CreativeCampaignResult> items;

        public List<CreativeCampaignResult> getItems() {
            return items;
        }

        public CampaignsListEntity setItems(List<CreativeCampaignResult> items) {
            this.items = items;
            return this;
        }
    }

    @GetMapping(value = "{id}/campaigns")
    public ResponseEntity<CampaignsListEntity> getCampaigns(@PathVariable(value = "id") String id,
                                                            @RequestParam("client_id") Long clientId) {
        authService.requirePermission(Privileges.Permission.CREATIVE_GET);

        List<CreativeCampaignResult> campaignsInfo;

        try {
            campaignsInfo = videoCreativesService.getCampaigns(id, authService.getUserId(), clientId);
        } catch (Exception e) {
            return ResponseEntity.ok(new CampaignsListEntity().setItems(Collections.emptyList()));
        }

        return ResponseEntity.ok(new CampaignsListEntity().setItems(campaignsInfo));
    }

}
