package ru.yandex.canvas.controllers.video;

import java.util.Collections;

import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.canvas.model.direct.Privileges;
import ru.yandex.canvas.model.video.Addition;
import ru.yandex.canvas.model.video.CustomVastParams;
import ru.yandex.canvas.model.video.PythiaAdditionExtraParams;
import ru.yandex.canvas.model.video.addition.AdditionData;
import ru.yandex.canvas.model.video.addition.AdditionDataBundle;
import ru.yandex.canvas.model.video.addition.AdditionElement;
import ru.yandex.canvas.model.video.addition.options.AdditionElementOptions;
import ru.yandex.canvas.service.AuthService;
import ru.yandex.canvas.service.SessionParams;

import static ru.yandex.canvas.service.video.VAST.SURVEY_CONFIG_MACROS;

@RestController
@RequestMapping(path = "/video/additions/survey")
public class VideoAdditionsSurveyController {

    public static final long DEFAULT_SURVEY_PRESET_ID = 6; // use cpm preset for now
    public static final String SURVEY_THEME_NAME = "video-banner_survey";
    public static final String DEFAULT_NAME = "SurveyCreative";

    private final VideoAdditionCreatorService videoAdditionCreatorService;
    private final AuthService authService;
    private final SessionParams sessionParams;

    public VideoAdditionsSurveyController(VideoAdditionCreatorService videoAdditionCreatorService,
            AuthService authService, SessionParams sessionParams)
    {
        this.videoAdditionCreatorService = videoAdditionCreatorService;
        this.authService = authService;
        this.sessionParams = sessionParams;
    }

    @PostMapping(path = "")
    public ResponseEntity<Addition> createAdditionSurvey(
            @RequestParam("client_id") Long clientId,
            @RequestBody PythiaAdditionExtraParams pythiaAdditionExtraParams
    )
    {
        authService.requirePermission(Privileges.Permission.CREATIVE_CREATE);
        sessionParams.setSessionType(SessionParams.SessionTag.CPM_BANNER);

        Addition addition = makeAddition(pythiaAdditionExtraParams);

        videoAdditionCreatorService.createAddition(authService.getUserId(), clientId, addition,
                new CustomVastParams().setVpaidPcodeUrl(pythiaAdditionExtraParams.getVpaidPcodeUrl()));

        return ResponseEntity.ok(addition);
    }

    private static Addition makeAddition(PythiaAdditionExtraParams pythiaAdditionExtraParams) {
        pythiaAdditionExtraParams.getPythia().setSurveyConfig(SURVEY_CONFIG_MACROS);
        // Use defaults for compatibility reasons
        Long presetId = pythiaAdditionExtraParams.getPresetId();
        if (null == presetId) {
            presetId = DEFAULT_SURVEY_PRESET_ID;
        }

        String name = pythiaAdditionExtraParams.getName();
        if (null == name) {
            name = DEFAULT_NAME;
        }

        return new Addition().setPresetId(presetId)
                .setName(name)
                .setPythia(pythiaAdditionExtraParams.getPythia())
                .setSkipUrl(pythiaAdditionExtraParams.getSkipUrl())
                .setVpaidPcodeUrl(pythiaAdditionExtraParams.getVpaidPcodeUrl())
                .setData(new AdditionData()
                        .setBundle(new AdditionDataBundle().setName(SURVEY_THEME_NAME))
                        .setElements(Collections.singletonList(new AdditionElement()
                                .withType(AdditionElement.ElementType.ADDITION)
                                .withOptions(
                                        new AdditionElementOptions().setVideoId(pythiaAdditionExtraParams.getVideoId())))));
    }
}
