package ru.yandex.canvas.controllers.video;

import java.io.IOException;
import java.net.URI;
import java.util.Set;

import javax.servlet.http.HttpServletRequest;
import javax.validation.Valid;
import javax.validation.constraints.NotNull;
import javax.validation.constraints.Positive;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import org.hibernate.validator.constraints.URL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.client.HttpStatusCodeException;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;
import org.springframework.web.multipart.MultipartFile;

import ru.yandex.canvas.configs.auth.AuthorizeBy;
import ru.yandex.canvas.controllers.video.wrappers.AudioFileWrapper;
import ru.yandex.canvas.controllers.video.wrappers.FileWrapper;
import ru.yandex.canvas.controllers.video.wrappers.InBannerWrapper;
import ru.yandex.canvas.controllers.video.wrappers.PackshotFileWrapper;
import ru.yandex.canvas.controllers.video.wrappers.VideoFileWrapper;
import ru.yandex.canvas.exceptions.BadRequestException;
import ru.yandex.canvas.exceptions.InternalServerError;
import ru.yandex.canvas.exceptions.VideoIntegrityValidationException;
import ru.yandex.canvas.model.ValidationError;
import ru.yandex.canvas.model.direct.Privileges;
import ru.yandex.canvas.model.video.AudioFiles;
import ru.yandex.canvas.model.video.CleanupOldConversionsResult;
import ru.yandex.canvas.model.video.files.AudioSource;
import ru.yandex.canvas.model.video.files.InBannerVideo;
import ru.yandex.canvas.model.video.files.Movie;
import ru.yandex.canvas.service.AuthService;
import ru.yandex.canvas.service.DirectService;
import ru.yandex.canvas.service.SandBoxService;
import ru.yandex.canvas.service.SessionParams;
import ru.yandex.canvas.service.TankerKeySet;
import ru.yandex.canvas.service.video.AudioServiceInterface;
import ru.yandex.canvas.service.video.HttpUtils;
import ru.yandex.canvas.service.video.InBannerVideoFilesService;
import ru.yandex.canvas.service.video.MovieServiceInterface;
import ru.yandex.canvas.service.video.PackshotServiceInterface;
import ru.yandex.canvas.service.video.VideoCreativeType;
import ru.yandex.canvas.service.video.VideoCreativesService;
import ru.yandex.canvas.service.video.VideoPresetsService;

import static ru.yandex.canvas.configs.auth.AuthorizeBy.AuthType.SANDBOX_CLEANUP_TOKEN;
import static ru.yandex.canvas.configs.auth.AuthorizeBy.AuthType.SANDBOX_SECRET;

@RestController
@RequestMapping(path = "/video/files")
public class VideoFilesModifyingController {
    private static final Logger logger = LoggerFactory.getLogger(VideoFilesModifyingController.class);
    private final AuthService authService;
    private final MovieServiceInterface movieService;
    private final PackshotServiceInterface packshotService;
    private final AudioServiceInterface audioService;
    private final SessionParams sessionParams;
    private final VideoCreativesService videoCreativesService;
    private final DirectService directService;
    private final InBannerVideoFilesService inBannerVideoFilesService;
    private final VideoPresetsService videoPresetsService;

    public VideoFilesModifyingController(AuthService authService,
                                         MovieServiceInterface movieService,
                                         AudioServiceInterface audioService,
                                         PackshotServiceInterface packshotService,
                                         SessionParams sessionParams,
                                         VideoCreativesService videoCreativesService,
                                         DirectService directService,
                                         InBannerVideoFilesService inBannerVideoFilesService,
                                         VideoPresetsService videoPresetsService) {

        this.authService = authService;
        this.movieService = movieService;
        this.audioService = audioService;
        this.packshotService = packshotService;
        this.sessionParams = sessionParams;
        this.videoCreativesService = videoCreativesService;
        this.directService = directService;
        this.inBannerVideoFilesService = inBannerVideoFilesService;
        this.videoPresetsService = videoPresetsService;
    }

    @PostMapping(path = "")
    public ResponseEntity<FileWrapper> uploadFile(
            @RequestParam("file") final MultipartFile file,
            @RequestParam(value = "type", required = false) String type,
            @RequestParam(value = "preset_id", required = false) Long presetId,
            @RequestParam("client_id") Long clientId) {
        authService.requirePermission(Privileges.Permission.CREATIVE_CREATE);
        VideoCreativeType videoCreativeType = sessionParams.getCreativeType();

        try {
            FileWrapper fileWrapper;

            if ("packshot".equals(type)) {
                fileWrapper = new PackshotFileWrapper(
                        packshotService.upload(file.getBytes(), file.getOriginalFilename(), clientId,
                                videoCreativeType, presetId));
            } else if ("audio".equals(type)) {
                AudioSource uploadedFile = audioService.upload(file.getBytes(), file.getOriginalFilename(),
                        clientId, videoCreativeType, presetId);
                fileWrapper = new AudioFileWrapper(uploadedFile);
            } else if ("in_banner".equals(type)) {
                InBannerVideo inBannerVideo = inBannerVideoFilesService.upload(file.getBytes(),
                        file.getOriginalFilename(), clientId, VideoCreativeType.IN_BANNER, presetId);
                fileWrapper = new InBannerWrapper(inBannerVideo);
            } else {

                if (presetId != null) {
                    videoCreativeType = videoPresetsService.fromPresetId(presetId);
                }

                Movie movie = movieService.upload(file.getBytes(), file.getOriginalFilename(), clientId,
                        videoCreativeType, presetId);

                Set<String> features = directService.getFeatures(clientId, null);

                fileWrapper = new VideoFileWrapper(movie, videoCreativeType, features);
            }

            return ResponseEntity.status(HttpStatus.CREATED)
                    .body(fileWrapper);

        } catch (HttpStatusCodeException e) {
            var exceptionText = e.getResponseBodyAsString();
            logger.error("File uploading to stillage failed: {}, {}", e.getStatusCode(), exceptionText);
            if (exceptionText.contains("Video integrity validation failed")) {
                // DIRECT-123887
                throw new VideoIntegrityValidationException(
                        TankerKeySet.VIDEO_VALIDATION_MESSAGES.key("video-integrity-validation-failed"), e);
            }
            throw new InternalServerError(e);
        } catch (IOException e) {
            throw new InternalServerError(e);
        }

    }

    public static class EventResponse {
        @JsonProperty("OK")
        Boolean ok;

        public Boolean getOk() {
            return ok;
        }

        public EventResponse setOk(Boolean ok) {
            this.ok = ok;
            return this;
        }
    }

    /*
    {'status': 'VIDEO_FAST_UPLOADED', 'stream_url': 'https://strm.yandex
    .ru/vh-canvas-converted/get-canvas/video_5c4edc6b78cd6545ad7cd257.m3u8', 'task_id': 368944578, 'formats':
    [{'url': 'https://strm.yandex.ru/vh-canvas-converted/get-canvas/video_5c4edc6b78cd6545ad7cd257_169_240p.mp4',
    'bitrate': 421, 'height': '240', 'width': '432', 'type': 'video/mp4', 'id':
    'video_5c4edc6b78cd6545ad7cd257_169_240p.mp4'}, {'url': 'https://strm.yandex
    .ru/vh-canvas-converted/get-canvas/video_5c4edc6b78cd6545ad7cd257_169_240p.webm', 'bitrate': 410, 'height':
    '240', 'width': '432', 'type': 'video/webm', 'id': 'video_5c4edc6b78cd6545ad7cd257_169_240p.webm'}, {'url':
    'https://strm.yandex.ru/vh-canvas-converted/get-canvas/video_5c4edc6b78cd6545ad7cd257_169_240p.ogv', 'bitrate':
    1261, 'height': '240', 'width': '432', 'type': 'video/ogg', 'id': 'video_5c4edc6b78cd6545ad7cd257_169_240p.ogv'}]}

    {'status': 'THUMBNAILS_UPLOADED', 'task_id': 368971541, 'thumbnail_urls':
    '["https://strm.yandex.ru/vh-canvas-converted/get_canvas/video_5c4b2ddb3bbdd16d033271fe_001.jpg"]'}

     */

    @AuthorizeBy({SANDBOX_SECRET})
    @PostMapping(path = "{id}/event")
    public ResponseEntity<EventResponse> registerUploadEvent(
            @PathVariable("id") String fileId,
            @RequestBody SandBoxService.SandboxConversionTaskOutput request
    ) {
        try {
            movieService.updateConvertingFile(fileId, request);
            videoCreativesService.patchVideoAddition(fileId);
        } catch (Exception e) {
            logger.error("Update failed {}", fileId);
            throw new InternalServerError(e);
        }

        return ResponseEntity.ok(new EventResponse().setOk(true));
    }

    @AuthorizeBy({SANDBOX_SECRET})
    @PostMapping(path = "{id}/ffhook")
    public ResponseEntity<EventResponse> registerFfEvent(
            @PathVariable("id") String fileId,
            @RequestBody SandBoxService.SandboxFfConversionTaskOutput request
    ) {
        logger.info("ff conversation task {} finish", request.getTaskId());
        AudioFiles file = audioService.updateConvertingFile(fileId, request);
        videoCreativesService.patchCpmAudioAddition(file);
        return ResponseEntity.ok(new EventResponse().setOk(true));
    }

    public ResponseEntity route() throws IOException {
        RequestAttributes attribs = RequestContextHolder.getRequestAttributes();
        HttpServletRequest request = ((ServletRequestAttributes) attribs).getRequest();
        return PythonRedirect.route(request);
    }

    public static class UploadUrlRequest {
        @NotNull
        @URL
        private String url;

        public String getUrl() {
            return url;
        }

        public UploadUrlRequest setUrl(String url) {
            this.url = url;
            return this;
        }
    }

    @PostMapping(path = "/url")
    public ResponseEntity<?> uploadFileFromUrl(
            @RequestBody @Valid UploadUrlRequest url,
            final BindingResult bindingResult,
            @RequestParam(value = "type", required = false) String type,
            @RequestParam(value = "preset_id", required = false) Long presetId,
            @RequestParam(value = "filename", required = false) String filename,
            @RequestParam("client_id") Long clientId
    ) {
        authService.requirePermission(Privileges.Permission.CREATIVE_CREATE);
        VideoCreativeType videoCreativeType = sessionParams.getCreativeType();

        if (bindingResult.hasErrors()) {
            logger.warn(bindingResult.getAllErrors().toString());
            return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(new ValidationError(
                    TankerKeySet.VIDEO_VALIDATION_MESSAGES.interpolate("not_an_url"),
                    bindingResult));
        }
        if (url.getUrl().isEmpty()) {
            return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(new ValidationError(
                    TankerKeySet.VIDEO_VALIDATION_MESSAGES.interpolate("empty_url"),
                    bindingResult));
        }

        URI uri = HttpUtils.makeURI(url.getUrl());
        // если название для файла не передано, то генерируем из url
        if (filename == null || filename.isEmpty()) {
            filename = HttpUtils.makeFilenameFromUrl(uri);
        }

        try {
            FileWrapper fileWrapper;

            if ("packshot".equals(type)) {
                fileWrapper = new PackshotFileWrapper(
                        packshotService.upload(uri, filename, clientId, videoCreativeType, presetId));
            } else if ("audio".equals(type)) {
                AudioSource uploadedFile = audioService.upload(uri, filename, clientId, videoCreativeType, presetId);
                fileWrapper = new AudioFileWrapper(uploadedFile);
            } else if ("in_banner".equals(type)) {
                InBannerVideo inBannerVideo = inBannerVideoFilesService.upload(uri,
                        filename, clientId, VideoCreativeType.IN_BANNER, presetId);
                fileWrapper = new InBannerWrapper(inBannerVideo);
            } else {
                Movie movie = movieService.upload(uri, filename, clientId, videoCreativeType, presetId);
                Set<String> features = directService.getFeatures(clientId, null);
                fileWrapper = new VideoFileWrapper(movie, videoCreativeType, features);
            }

            return ResponseEntity.status(HttpStatus.CREATED).body(fileWrapper);

        } catch (HttpStatusCodeException e) {
            logger.error(String.format("File uploading to stillage failed: %s, %s",
                    e.getStatusCode(), e.getResponseBodyAsString()), e);

            if (e.getStatusCode() == HttpStatus.PAYLOAD_TOO_LARGE) {
                return ResponseEntity.status(HttpStatus.PAYLOAD_TOO_LARGE).body(null);
            } else if (e.getStatusCode().is4xxClientError() || e.getStatusCode().is5xxServerError()) {
                throw new BadRequestException("Provided file url is unavailable");
            } else {
                throw new InternalServerError(e);
            }

        } catch (IOException e) {
            throw new InternalServerError(e);
        }

    }

    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class CleanupRequest {
        @JsonProperty("age_seconds")
        @Positive
        private Long ageSeconds;

        public Long getAgeSeconds() {
            return ageSeconds;
        }

        public CleanupRequest setAgeSeconds(Long ageSeconds) {
            this.ageSeconds = ageSeconds;
            return this;
        }
    }

    public static class CleanupResponse {
        @JsonProperty("OK")
        private Boolean ok;

        @JsonProperty("missing_sandbox_tasks")
        private Long missingSandboxTasks;

        public CleanupResponse(Boolean ok, Long missingSandboxTasks) {
            this.ok = ok;
            this.missingSandboxTasks = missingSandboxTasks;
        }

        public Boolean getOk() {
            return ok;
        }

        public Long getMissingSandboxTasks() {
            return missingSandboxTasks;
        }
    }

    @AuthorizeBy({SANDBOX_CLEANUP_TOKEN})
    @PostMapping(path = "/tasks/cleanup")
    public ResponseEntity<CleanupResponse> cleanupTasks(@RequestBody(required = false) CleanupRequest cleanupRequest) {
        long ageSeconds = 300;

        if (cleanupRequest != null && cleanupRequest.getAgeSeconds() != null) {
            ageSeconds = cleanupRequest.getAgeSeconds();
        }
        logger.info("cleanupTasks, ageSeconds={}", ageSeconds);

        CleanupOldConversionsResult cleanupResult = movieService.cleanupOldConversions(ageSeconds);
        // переотправляем креативы с только что сконвертированными видео в rtb
        cleanupResult.getReadyFileIds().forEach(videoCreativesService::patchVideoAddition);

        //проверяем подвисшие конвертации CMS
        videoCreativesService.cleanupCmsConversions(ageSeconds);

        return ResponseEntity.ok(new CleanupResponse(true, cleanupResult.getMissedSandboxTasksCount()));
    }
}
