package ru.yandex.canvas.controllers.video;

import java.util.Comparator;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import javax.validation.Valid;
import javax.validation.constraints.Min;

import com.fasterxml.jackson.annotation.JsonProperty;
import org.jetbrains.annotations.NotNull;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.canvas.exceptions.NotFoundException;
import ru.yandex.canvas.service.DirectService;
import ru.yandex.canvas.service.SessionParams;
import ru.yandex.canvas.service.VideoLimitsInterface;
import ru.yandex.canvas.service.video.VideoCreativeType;
import ru.yandex.canvas.service.video.VideoLimitsService;
import ru.yandex.canvas.service.video.VideoPresetsService;
import ru.yandex.canvas.service.video.presets.VideoPreset;

import static ru.yandex.canvas.VideoConstants.UNWANTED_PRESET_IDS;
import static ru.yandex.direct.feature.FeatureName.HIDE_UNWANTED_PRESET;
import static ru.yandex.direct.utils.CommonUtils.nvl;


@RestController
@RequestMapping(path = "/video/presets")
@Validated
public class VideoPresetsController {

    private final VideoPresetsService videoPresetsService;
    private final SessionParams sessionParams;
    private final VideoLimitsService videoLimitsService;
    private final DirectService directService;

    public VideoPresetsController(VideoPresetsService videoPresetsService, SessionParams sessionParams,
                                  VideoLimitsService videoLimitsService, DirectService directService) {
        this.videoPresetsService = videoPresetsService;
        this.sessionParams = sessionParams;
        this.videoLimitsService = videoLimitsService;
        this.directService = directService;
    }

    @GetMapping(path = "{id}")
    public PresetWrapper getPreset(@PathVariable("id") @Valid @Min(1) Long id) {
        VideoPreset videoPreset = videoPresetsService.getPreset(id);
        Set<String> features = directService.getFeatures(sessionParams.getClientId(), null);
        VideoCreativeType creativeType = videoPresetsService.fromPresetId(videoPreset.getId());
        var limits = videoLimitsService.getLimits(creativeType, videoPreset.getId());

        if (videoPreset == null) {
            throw new NotFoundException();
        }
        return new PresetWrapper(videoPreset,
                tooltip -> videoPresetsService.tooltipInterpolator(tooltip, creativeType, videoPreset.getId()),
                limits, features);
    }

    public static class PresetsList {
        @JsonProperty("total")
        Integer total;

        @JsonProperty("items")
        List<PresetWrapper> presetWrappers;


        public Integer getTotal() {
            return total;
        }

        public List<PresetWrapper> getPresetWrappers() {
            return presetWrappers;
        }
    }

    //http://172.17.42.1:11362/video/presets?cpc_video_banner=1&cpm_banner=&mobile_content_banner=&user_id=734948687
    // &client_id=103997791
    @GetMapping(path = "")
    public PresetsList getPresetsByTag() {
        PresetsList presetsList = new PresetsList();
        Set<String> features = directService.getFeatures(sessionParams.getClientId(), null);

        if (sessionParams.getVideoPresetIds() != null) {
            presetsList.presetWrappers = sessionParams.getVideoPresetIds().stream()
                    .peek(e -> {
                        if (!videoPresetsService.contains(e)) {
                            throw new NotFoundException();
                        }
                    })
                    .map(videoPresetsService::getPreset)
                    .filter(e -> presetFilter(e, features))
                    .sorted(Comparator.comparing(VideoPreset::getOrder))
                    .map(e -> getPresetWrapper(videoLimitsService.getLimits(sessionParams.getCreativeType(), e.getId()),
                            e, features))
                    .collect(Collectors.toList());
        } else {
            presetsList.presetWrappers =
                    videoPresetsService.getPresetsByCreativeType(sessionParams.getCreativeType())
                            .stream()
                            .filter(e -> presetFilter(e, features))
                            .sorted(Comparator.comparing(VideoPreset::getOrder))
                            .map(e -> getPresetWrapper(videoLimitsService.getLimits(sessionParams.getCreativeType(), e.getId()),
                                    e, features))
                            .collect(Collectors.toList());
        }

        presetsList.total = presetsList.presetWrappers.size();

        return presetsList;
    }

    @NotNull
    private PresetWrapper getPresetWrapper(VideoLimitsInterface limits, VideoPreset e, Set<String> features) {
        return new PresetWrapper(e,
                tooltip -> videoPresetsService.tooltipInterpolator(tooltip, videoPresetsService.fromPresetId(e.getId()), e.getId()),
                limits, features);
    }

    private static boolean presetFilter(VideoPreset preset, Set<String> features) {
        return !nvl(preset.getDescription().getHiddenFromList(), false)
                && (!features.contains(HIDE_UNWANTED_PRESET.getName())
                    || !UNWANTED_PRESET_IDS.contains(preset.getId()));
    }
}
