package ru.yandex.canvas.controllers.video.wrappers;

import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;

import ru.yandex.canvas.VideoConstants;
import ru.yandex.canvas.controllers.video.DateSerializer;
import ru.yandex.canvas.model.video.files.FileStatus;
import ru.yandex.canvas.model.video.files.Movie;
import ru.yandex.canvas.model.video.files.Thumbnail;
import ru.yandex.canvas.service.TankerKeySet;
import ru.yandex.canvas.service.video.VideoCreativeType;

import static java.util.Collections.emptyList;

public class VideoFileWrapper implements FileWrapper {
    @JsonIgnore
    private final Movie movie;
    @Nullable
    @JsonIgnore
    private final VideoCreativeType videoCreativeType;
    @Nullable
    @JsonIgnore
    private final Set<String> features;

    @Nullable
    @JsonProperty("preset_id")
    @JsonInclude(JsonInclude.Include.NON_NULL)
    private final Long presetId;

    public VideoFileWrapper(Movie movie) {
        this(movie, null, null);
    }

    public VideoFileWrapper(Movie movie,
                            @Nullable VideoCreativeType videoCreativeType,
                            @Nullable Set<String> features) {
        this(movie, null, videoCreativeType, features);
    }

    public VideoFileWrapper(Movie movie,
                            Long presetId,
                            @Nullable VideoCreativeType videoCreativeType,
                            @Nullable Set<String> features) {
        this.movie = movie;
        this.presetId = presetId;
        this.videoCreativeType = videoCreativeType;
        this.features = features;
    }

    @JsonProperty("archive")
    public Boolean getArchive() {
        return movie.getArchive();
    }

    @JsonProperty("id")
    public String getId() {
        return movie.getVideoSource().getId();
    }

    @JsonProperty("name")
    public String getName() {
        return movie.getVideoSource().getName();
    }

    @JsonProperty("formats")
    public List<VideoFileFormatRecord> getFormat() {
        if (movie.getVideoSource().getStockId() != null) {
            return Arrays.asList(new VideoFileFormatRecord(
                    movie.getVideoSource().getStillageUrl(),
                    null, null, null)); //Why stillage url? Why not any url from formats?
        }

        return movie.getFormats().stream()
                .map(e -> new VideoFileFormatRecord(e.getUrl(), e.getBitrate(), e.getDelivery(), e.getMimeType()))
                .collect(Collectors.toList());
    }

    @JsonProperty("status")
    public FileStatus getStatus() {
        return movie.getStatus();
    }

    @JsonProperty("show_early_preview")
    public Boolean showEarlyPreview() {
        return movie.isShowEarlyPreviewAllowed();
    }

    @JsonProperty("create_early_creative")
    public Boolean createEarlyCreative() {
        return movie.isCreateEarlyCreativeAllowed(videoCreativeType, features);
    }


    @JsonProperty("stock_file_id")
    public String getStockFileId() {
        return movie.getVideoSource().getStockId();
    }

    //        @JsonInclude(JsonInclude.Include.ALWAYS)
    @JsonProperty("conversion_task_id")
    public Long getConversionTaskId() {
        return movie.getConvertionTaskId();
    }

    @JsonProperty("overlayColor")
    public String getOverlayColor() {
        return movie.getVideoSource().getOverlayColor();
    }

    @JsonProperty("stillage_id")
    public String getStillageId() {
        return movie.getVideoSource().getStillageId();
    }

    @JsonProperty("type")
    public String getType() {
        return "video";
    }

    @JsonProperty("date")
    @JsonSerialize(using = DateSerializer.class)
    public Date getDate() {
        return movie.getVideoSource().getDate();
    }

    @JsonProperty("client_id")
    public Long clientId() {
        return movie.getVideoSource().getClientId();
    }

    @JsonProperty("url")
    public String getUrl() {
        return movie.getVideoSource().getStillageUrl();
    }

    @JsonProperty("thumbnail")
    public Thumbnail getThumbnail() {
        return movie.getVideoSource().getThumbnail();
    }

    @JsonProperty("thumbnailUrl")
    public String getThumbnailUrl() {

        if (movie.getVideoSource().getThumbnail() == null
                || movie.getVideoSource().getThumbnail().getUrl() == null) {
            return VideoConstants.DEFAULT_THUMBNAIL_URL;
        }

        return movie.getVideoSource().getThumbnail().getUrl();
    }

    @JsonProperty("error_message")
    public String getErrorMessage() {
        if (movie.getStatus() == FileStatus.ERROR) {
            return TankerKeySet.VIDEO.key("failed_to_convert_file");
        }

        return null;
    }

    public static class VideoFileFormatRecord {
        private String url;
        private Long bitrate;
        private String delivery;
        private String type;

        public VideoFileFormatRecord(String url, Long bitrate, String delivery, String type) {
            this.url = url;
            this.bitrate = bitrate;
            this.delivery = delivery;
            this.type = type;
        }

        public String getType() {
            return type;
        }

        public String getUrl() {
            return url;
        }

        public Long getBitrate() {
            return bitrate;
        }

        public String getDelivery() {
            return delivery;
        }
    }

    @JsonProperty("sub_categories")
    public List<Long> getCategories() {
        if (movie.getVideoSource().getCategoryId() == null || movie.getVideoSource().getCategoryId().equals("0")) {
            return emptyList();
        }

        return Arrays.asList(Long.parseLong(movie.getVideoSource().getCategoryId()));
    }

    @JsonProperty("width")
    public Integer getWidth() {
        return movie.getVideoSource().getWidth();
    }

    @JsonProperty("height")
    public Integer getHeight() {
        return movie.getVideoSource().getHeight();
    }

    @JsonProperty("duration")
    public Double getDuration() {
        return movie.getDuration();
    }

    @JsonProperty("mime_type")
    public String getMimeType() {
        return movie.getVideoSource().getMimeType();
    }
}
