package ru.yandex.canvas.model;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.validation.Valid;
import javax.validation.constraints.NotNull;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonInclude;
import org.springframework.beans.BeanUtils;

import ru.yandex.canvas.model.elements.Element;
import ru.yandex.canvas.model.elements.ElementType;
import ru.yandex.canvas.model.validation.DifferentlyColoredOptions;
import ru.yandex.canvas.model.validation.presetbased.creative.AllowedDomainsValidator;
import ru.yandex.canvas.model.validation.presetbased.creative.CreativeDataValidator;
import ru.yandex.canvas.model.validation.presetbased.creative.CroppedImageValidator;
import ru.yandex.canvas.model.validation.presetbased.creative.ImageFileSizeValidator;
import ru.yandex.canvas.model.validation.presetbased.creative.ImageSizeValidator;
import ru.yandex.canvas.model.validation.presetbased.creative.ValidCreativeColorValidator;
import ru.yandex.canvas.model.validation.presetbased.creative.ValidImageValidator;
import ru.yandex.canvas.service.rtbhost.ConstructorData.ConstructorDataElement;

import static java.util.stream.Collectors.toList;
import static ru.yandex.canvas.model.validation.DifferentlyColoredOptions.Metric.CIEDE2000;
import static ru.yandex.direct.utils.CommonUtils.nvl;

/**
 * Actual creative data used in construction.
 *
 * @author pupssman
 */
public class CreativeData extends CreativeDimension<CreativeData> {

    @Valid
    private Options options;
    @Valid
    @NotNull
    private List<Element> elements;
    @Valid
    @NotNull
    private Map<String, MediaSet> mediaSets;
    @NotNull
    @Valid
    private Bundle bundle;
    private String clickUrl;
    private List<CreativeDataValidator> validators;

    public Options getOptions() {
        return options;
    }

    public void setOptions(Options options) {
        this.options = options;
    }

    public List<Element> getElements() {
        return elements;
    }

    public void setElements(List<Element> elements) {
        this.elements = elements;
    }

    public Bundle getBundle() {
        return bundle;
    }

    public void setBundle(Bundle bundle) {
        this.bundle = bundle;
    }

    public String getClickUrl() {
        return clickUrl;
    }

    public void setClickUrl(String clickUrl) {
        this.clickUrl = clickUrl;
    }

    public Map<String, MediaSet> getMediaSets() {
        return mediaSets;
    }

    public void setMediaSets(Map<String, MediaSet> mediaSets) {
        this.mediaSets = mediaSets;
    }

    public CreativeData withOptions(Options options) {
        this.options = options;
        return this;
    }

    public void setValidators(List<CreativeDataValidator> validators) {
        this.validators = validators;
    }

    @JsonIgnore
    public List<CreativeDataValidator> getEffectiveValidators() {
        return nvl(validators, getDefaultValidators());
    }

    /**
     * Validations to be used to validate element, if none are present in preset configuration
     */
    @JsonIgnore
    protected List<CreativeDataValidator> getDefaultValidators() {
        return List.of(
                new ValidCreativeColorValidator(),
                new ValidImageValidator(),
                new AllowedDomainsValidator(),
                new CroppedImageValidator(List.of(ElementType.IMAGE, ElementType.SECOND_IMAGE)),
                new ImageFileSizeValidator(512000),
                new ImageSizeValidator(Collections.singletonList(ElementType.LOGO), 300, 300, 40 * 1024)
        );
    }

    /**
     * @return a light copy of <b>this</b> with all the hidden elements removed;
     * <p>
     * Use it when passing values to other systems (like HTML generation and moderation data export)
     */
    @JsonIgnore
    public CreativeData onlyVisibleData() {
        CreativeData result = new CreativeData();

        BeanUtils.copyProperties(this, result, "elements", "mediaSets");
        result.setElements(elements.stream().filter(Element::getAvailable).collect(toList()));

        // Take only mediaSets mentioned in visible elements
        Set<String> usedMediaSets = result.getElements().stream()
                .map(Element::getMediaSet)
                .filter(Objects::nonNull)
                .collect(Collectors.toSet());

        result.setMediaSets(mediaSets.entrySet().stream()
                .filter(e -> usedMediaSets.contains(e.getKey()))
                .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue)));

        return result;
    }

    @JsonIgnore
    public List<ConstructorDataElement> toRTBHostExportData() {
        return elements.stream()
                .map(element -> element.toRTBHostExportData(mediaSets))
                .filter(Objects::nonNull)
                .collect(toList());
    }

    @Override
    public String toString() {
        return "CreativeData{" +
                "options=" + options +
                ", elements=" + elements +
                ", mediaSets=" + mediaSets +
                ", bundle=" + bundle +
                ", clickUrl='" + clickUrl + '\'' +
                '}';
    }

    @DifferentlyColoredOptions(distance = 9.8, metric = CIEDE2000)
    public static class Options {

        private String backgroundColor;

        private String borderColor;

        /**
         * Whether the ad has animation
         */
        private Boolean hasAnimation;


        private Boolean hasSocialLabel;

        /**
         * True if ad is adaptive (one images and element-set for all sizes)
         */
        private Boolean isAdaptive;

        /**
         * Минимальные размеры креатива (только для адаптивых ГО)
         */
        @JsonInclude(JsonInclude.Include.NON_NULL)
        private Integer minWidth;
        @JsonInclude(JsonInclude.Include.NON_NULL)
        private Integer minHeight;

        public String getBackgroundColor() {
            return backgroundColor;
        }

        public void setBackgroundColor(String backgroundColor) {
            this.backgroundColor = backgroundColor;
        }

        public String getBorderColor() {
            return borderColor;
        }

        public void setBorderColor(String borderColor) {
            this.borderColor = borderColor;
        }

        public Boolean getHasAnimation() {
            return hasAnimation;
        }

        public void setHasAnimation(Boolean hasAnimation) {
            this.hasAnimation = hasAnimation;
        }

        public Boolean getIsAdaptive() {
            return isAdaptive != null ? isAdaptive : false;
        }

        public void setIsAdaptive(Boolean isAdaptive) {
            this.isAdaptive = isAdaptive;
        }

        public Integer getMinWidth() {
            return minWidth;
        }

        public void setMinWidth(Integer minWidth) {
            this.minWidth = minWidth;
        }

        public Integer getMinHeight() {
            return minHeight;
        }

        public void setMinHeight(Integer minHeight) {
            this.minHeight = minHeight;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            Options options = (Options) o;
            return Objects.equals(backgroundColor, options.backgroundColor) &&
                    Objects.equals(borderColor, options.borderColor) &&
                    Objects.equals(hasAnimation, options.hasAnimation) &&
                    Objects.equals(hasSocialLabel, options.hasSocialLabel) &&
                    Objects.equals(isAdaptive, options.isAdaptive) &&
                    Objects.equals(minWidth, options.minWidth) &&
                    Objects.equals(minHeight, options.minHeight);
        }

        @Override
        public int hashCode() {
            return Objects.hash(backgroundColor, borderColor, hasAnimation, hasSocialLabel, isAdaptive, minWidth, minHeight);
        }

        @Override
        public String toString() {
            return "Options{" +
                    "backgroundColor='" + backgroundColor + '\'' +
                    ", borderColor='" + borderColor + '\'' +
                    ", hasAnimation=" + hasAnimation +
                    ", hasSocialLabel=" + hasSocialLabel +
                    ", isAdaptive=" + isAdaptive +
                    ", minWidth=" + minWidth +
                    ", minHeight=" + minHeight +
                    '}';
        }

        public Boolean getHasSocialLabel() {
            return hasSocialLabel;
        }

        public void setHasSocialLabel(Boolean hasSocialLabel) {
            this.hasSocialLabel = hasSocialLabel;
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        CreativeData that = (CreativeData) o;

        if (!Objects.equals(options, that.options)) {
            return false;
        }
        if (!Objects.equals(elements, that.elements)) {
            return false;
        }
        if (!Objects.equals(mediaSets, that.mediaSets)) {
            return false;
        }
        if (!Objects.equals(bundle, that.bundle)) {
            return false;
        }
        return Objects.equals(clickUrl, that.clickUrl);
    }

    @Override
    public int hashCode() {
        int result = options != null ? options.hashCode() : 0;
        result = 31 * result + (elements != null ? elements.hashCode() : 0);
        result = 31 * result + (mediaSets != null ? mediaSets.hashCode() : 0);
        result = 31 * result + (bundle != null ? bundle.hashCode() : 0);
        result = 31 * result + (clickUrl != null ? clickUrl.hashCode() : 0);
        return result;
    }
}
