package ru.yandex.canvas.model;

import java.util.Date;
import java.util.Objects;

import javax.annotation.Nullable;
import javax.validation.Valid;
import javax.validation.constraints.NotNull;

import com.google.common.collect.ComparisonChain;
import org.springframework.data.annotation.Transient;

import ru.yandex.canvas.model.validation.PresetBasedValidation;

/**
 * @author skirsanov
 */
@PresetBasedValidation
public class CreativeDocument implements Comparable<CreativeDocument>, CreativeWithId, CreativeWithScreenshot,
        CreativeWithAdminRejectReason {

    private long id;
    private Boolean available;

    @Transient // stored in batch
    private String name;
    @Transient // stored in batch
    private Date date;
    @Transient // stored in batch
    private String batchId;
    @Transient // stored in batch
    private Long clientId;

    private int itemId;

    private String creativeURL;
    private String screenshotURL;
    private Boolean screenshotIsDone;

    @Transient // filled in CreativeService::creativeDocumentConsumer
    private String previewURL;

    @Valid
    @NotNull
    private CreativeData data;

    @Nullable
    private Integer presetId;

    @Nullable
    private String adminRejectReason;


    public CreativeDocument() {
    }


    public long getId() {
        return id;
    }

    @Override
    public Long fetchCreativeId() {
        return getId();
    }

    @Override
    public String fetchScreenshotUrl() {
        return getScreenshotURL();
    }

    @Override
    public Boolean fetchScreenshotIsDone() {
        return getScreenshotIsDone();
    }

    public void setId(long id) {
        this.id = id;
    }

    public CreativeData getData() {
        return data;
    }

    public void setData(CreativeData data) {
        this.data = data;
    }

    public String getCreativeURL() {
        return creativeURL;
    }

    public void setCreativeURL(String creativeURL) {
        this.creativeURL = creativeURL;
    }

    public String getScreenshotURL() {
        return screenshotURL;
    }

    public void setScreenshotURL(String screenshotURL) {
        this.screenshotURL = screenshotURL;
    }

    public Boolean getScreenshotIsDone() {
        return screenshotIsDone;
    }

    public void setScreenshotIsDone(Boolean screenshotIsDone) {
        this.screenshotIsDone = screenshotIsDone;
    }

    public String getName() {
        return name;
    }

    public void setName(String name) {
        this.name = name;
    }

    public Date getDate() {
        return date;
    }

    public void setDate(Date date) {
        this.date = date;
    }

    public String getBatchId() {
        return batchId;
    }

    public void setBatchId(String batchId) {
        this.batchId = batchId;
    }

    public Long getClientId() {
        return clientId;
    }

    public CreativeDocument setClientId(Long clientId) {
        this.clientId = clientId;
        return this;
    }

    public String getPreviewURL() {
        return previewURL;
    }

    public void setPreviewURL(String previewURL) {
        this.previewURL = previewURL;
    }

    public Integer getPresetId() {
        return presetId;
    }

    public CreativeDocument withId(long id) {
        this.id = id;
        return this;
    }

    public CreativeDocument withPresetId(@Nullable Integer presetId) {
        this.presetId = presetId;
        return this;
    }

    public CreativeDocument withAvailable(Boolean available) {
        this.available = available;
        return this;
    }

    public CreativeDocument withData(CreativeData data) {
        this.data = data;
        return this;
    }

    @Override
    public String toString() {
        return "CreativeDocument{" +
                "id=" + id +
                ", available=" + available +
                ", name='" + name + '\'' +
                ", date=" + date +
                ", batchId='" + batchId + '\'' +
                ", itemId=" + itemId +
                ", creativeURL='" + creativeURL + '\'' +
                ", screenshotURL='" + screenshotURL + '\'' +
                ", screenshotIsDone=" + screenshotIsDone +
                ", previewURL='" + previewURL + '\'' +
                ", data=" + data +
                ", presetId=" + presetId +
                ", adminRejectReason='" + adminRejectReason + '\'' +
                '}';
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        CreativeDocument that = (CreativeDocument) o;
        return id == that.id &&
                itemId == that.itemId &&
                Objects.equals(available, that.available) &&
                Objects.equals(name, that.name) &&
                Objects.equals(date, that.date) &&
                Objects.equals(batchId, that.batchId) &&
                Objects.equals(creativeURL, that.creativeURL) &&
                Objects.equals(screenshotURL, that.screenshotURL) &&
                Objects.equals(screenshotIsDone, that.screenshotIsDone) &&
                Objects.equals(previewURL, that.previewURL) &&
                Objects.equals(data, that.data) &&
                Objects.equals(presetId, that.presetId) &&
                Objects.equals(adminRejectReason, that.adminRejectReason);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, available, name, date, batchId, itemId, creativeURL, screenshotURL, screenshotIsDone,
                previewURL, data, presetId, adminRejectReason);
    }

    public int getItemId() {
        return itemId;
    }

    public CreativeDocument withItemId(int itemId) {
        this.itemId = itemId;
        return this;
    }

    @Nullable
    public String getAdminRejectReason() {
        return adminRejectReason;
    }

    public CreativeDocument withAdminRejectReason(@Nullable String adminRejectReason) {
        this.adminRejectReason = adminRejectReason;
        return this;
    }

    @NotNull
    public Boolean getAvailable() {
        return available == null ? true : available;
    }

    public void setAvailable(Boolean available) {
        this.available = available;
    }

    @Override
    public int compareTo(CreativeDocument that) {
        return ComparisonChain.start()
                .compareTrueFirst(getAvailable(), that.getAvailable())
                .compare(getData(), that.getData())
                .result();
    }

}
