package ru.yandex.canvas.model;

import java.util.Date;
import java.util.List;
import java.util.Objects;

import javax.validation.Valid;
import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import javax.validation.constraints.Size;

import com.fasterxml.jackson.annotation.JsonIgnore;
import org.springframework.data.annotation.Id;
import org.springframework.data.mongodb.core.index.CompoundIndex;
import org.springframework.data.mongodb.core.index.CompoundIndexes;
import org.springframework.data.mongodb.core.mapping.Document;

import ru.yandex.canvas.model.validation.PresetBasedValidation;

/**
 * @author skirsanov
 */
@Document(collection = "creative_batch")
@CompoundIndexes({
        @CompoundIndex(def = "{'items._id': 1, 'archive': 1}"),
        @CompoundIndex(def = "{'clientId': 1, 'date': 1, 'archive': 1}"),
        @CompoundIndex(def = "{'clientId': 1, 'date': -1, 'archive': 1}")
})
@PresetBasedValidation
public class CreativeDocumentBatch {

    private Long clientId;

    @Id
    private String id; //TODO use ObjectId type

    private String ideaId;

    @NotBlank
    private String name;

    //TODO:: check the type
    private Date date;

    private int total;

    @Valid
    @Size(min = 1)
    @NotNull
    private List<CreativeDocument> items;

    @JsonIgnore
    private boolean archive;


    public CreativeDocumentBatch() {
    }

    public int getTotal() {
        return total;
    }

    public void setTotal(int total) {
        this.total = total;
    }

    public Date getDate() {
        return date;
    }

    public void setDate(Date date) {
        this.date = date;
    }

    public String getId() {
        // Old batches have Double _id, new - ObjectId.
        // See BANNERSTORAGE-4727
        return id.length() < 24
                ? String.valueOf(Double.valueOf(id).intValue())
                : id;
    }

    public void setId(String id) {
        this.id = id;
    }

    public String getName() {
        return name;
    }

    public void setName(String name) {
        this.name = name;
    }

    public List<CreativeDocument> getItems() {
        return items;
    }

    public void setItems(List<CreativeDocument> items) {
        this.items = items;
    }


    public Long getClientId() {
        return clientId;
    }

    public void setClientId(Long clientId) {
        this.clientId = clientId;
    }

    public boolean isArchive() {
        return archive;
    }

    public void setArchive(boolean archive) {
        this.archive = archive;
    }

    public String getIdeaId() {
        return ideaId;
    }

    public void setIdeaId(String ideaId) {
        this.ideaId = ideaId;
    }

    public CreativeDocumentBatch withClientId(Long clientId) {
        this.clientId = clientId;
        return this;
    }

    public CreativeDocumentBatch withId(String id) {
        this.id = id;
        return this;
    }

    public CreativeDocumentBatch withIdeaId(String ideaId) {
        this.ideaId = ideaId;
        return this;
    }

    public CreativeDocumentBatch withName(String name) {
        this.name = name;
        return this;
    }

    public CreativeDocumentBatch withDate(Date date) {
        this.date = date;
        return this;
    }

    public CreativeDocumentBatch withTotal(int total) {
        this.total = total;
        return this;
    }

    public CreativeDocumentBatch withItems(List<CreativeDocument> items) {
        this.items = items;
        return this;
    }

    public CreativeDocumentBatch withArchive(boolean archive) {
        this.archive = archive;
        return this;
    }

    @Override
    public String toString() {
        return "CreativeDocumentBatch{" +
                "clientId=" + clientId +
                ", id='" + id + '\'' +
                ", ideaId='" + ideaId + '\'' +
                ", name='" + name + '\'' +
                ", date=" + date +
                ", total=" + total +
                ", items=" + items +
                ", archive=" + archive +
                '}';
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        CreativeDocumentBatch batch = (CreativeDocumentBatch) o;

        if (total != batch.total) {
            return false;
        }
        if (archive != batch.archive) {
            return false;
        }
        if (!Objects.equals(clientId, batch.clientId)) {
            return false;
        }
        if (!Objects.equals(id, batch.id)) {
            return false;
        }
        if (!Objects.equals(ideaId, batch.ideaId)) {
            return false;
        }
        if (!Objects.equals(name, batch.name)) {
            return false;
        }
        if (!Objects.equals(date, batch.date)) {
            return false;
        }
        return Objects.equals(items, batch.items);
    }

    @Override
    public int hashCode() {
        int result = clientId != null ? clientId.hashCode() : 0;
        result = 31 * result + (id != null ? id.hashCode() : 0);
        result = 31 * result + (ideaId != null ? ideaId.hashCode() : 0);
        result = 31 * result + (name != null ? name.hashCode() : 0);
        result = 31 * result + (date != null ? date.hashCode() : 0);
        result = 31 * result + total;
        result = 31 * result + (items != null ? items.hashCode() : 0);
        result = 31 * result + (archive ? 1 : 0);
        return result;
    }
}
