package ru.yandex.canvas.model;

import java.util.HashSet;
import java.util.Set;

import static ru.yandex.canvas.service.TankerKeySet.MIGRATION_MESSAGES;

public class CreativeMigrationResult {
    private boolean hasCropChanges = false;
    private Set<String> newSizes = new HashSet<>();
    private Set<String> newElements = new HashSet<>();

    public CreativeMigrationResult() {
    }

    public CreativeMigrationResult(boolean hasCropChanges) {
        this.hasCropChanges = hasCropChanges;
    }

    public boolean isHasCropChanges() {
        return hasCropChanges;
    }

    public void setHasCropChanges(boolean hasCropChanges) {
        this.hasCropChanges = hasCropChanges;
    }

    public Set<String> getNewSizes() {
        return newSizes;
    }

    public void setNewSizes(Set<String> newSizes) {
        this.newSizes = newSizes;
    }

    public void addNewSize(String size) {
        this.newSizes.add(size);
    }

    public Set<String> getNewElements() {
        return newElements;
    }

    public void setNewElements(Set<String> newElements) {
        this.newElements = newElements;
    }

    /**
     * Merges this result entry with another result entry returning a unified one.
     */
    public CreativeMigrationResult and(CreativeMigrationResult another) {
        this.hasCropChanges |= another.hasCropChanges;
        this.newSizes.addAll(another.getNewSizes());
        this.newElements.addAll(another.getNewElements());
        return this;
    }

    private String getLocalized(String key) {
        return MIGRATION_MESSAGES.key(key);
    }

    private String bulletPoint(String something) {
        return bulletPoint(something, ";");
    }

    private String bulletPoint(String something, String lineTerminator) {
        return " \u2022 " + something + lineTerminator + "\n";  // u2022 is the bullet point
    }

    @org.jetbrains.annotations.NotNull
    public String getMigrationMessage() {
        StringBuilder result = new StringBuilder();
        result.append(getLocalized("preset-updated"));

        if (isHasCropChanges()) {
            result.append(" ")
                    .append(getLocalized("extra-actions-required-to-continue"))
                    .append("\n")
                    .append(bulletPoint(getLocalized("crop-image-again"), "."))
                    .append(getLocalized("what-else-changed"));
        } else {
            result.append(getLocalized("what-changed"));
        }
        result.append("\n");

        // update bullet points start

        if (newElements.contains("logo")) {
            result.append(bulletPoint(getLocalized("new-element-logo")));
        }

        if (newElements.contains("domain")) {
            result.append(bulletPoint(getLocalized("new-element-domain")));
        }

        if (newElements.contains("fade")) {
            result.append(bulletPoint(getLocalized("new-element-fade")));
        }

        result.append(bulletPoint(getLocalized("longer-headline")));

        if (newSizes.isEmpty()) {
            // no new sizes -- generate last one with dot
            result.append(bulletPoint(getLocalized("longer-description"), "."));
        } else {
            // new sizes -- generate new sizes part with dot
            result.append(bulletPoint(getLocalized("longer-description")));
            result.append(bulletPoint(getLocalized("new-sizes")
                    + " (" + String.join(", ", getNewSizes()) + ")", "."));
        }

        return result.toString();
    }
}
