package ru.yandex.canvas.model;

import java.util.Date;
import java.util.Optional;

import com.fasterxml.jackson.annotation.JsonIgnore;
import org.jetbrains.annotations.Nullable;
import org.springframework.data.annotation.Id;
import org.springframework.data.mongodb.core.index.CompoundIndex;
import org.springframework.data.mongodb.core.index.CompoundIndexes;
import org.springframework.data.mongodb.core.mapping.Document;

import ru.yandex.canvas.model.stillage.StillageFileInfo;

/**
 * @author skirsanov
 */
@Document(collection = "files")
@CompoundIndexes({
        // FIXME: it's better to use here two partial unique indices, but I'm not sure that spring-data-mongodb supports it
        // Unique index to prevent duplicates in deduplication process, see BANNERSTORAGE-4800
        @CompoundIndex(def = "{'clientId': 1, 'userId': 1, 'stillageFileId': 1, 'name': 1}", unique = true),

        // indexes to speed up /files/my/ lookup
        @CompoundIndex(def = "{'clientId': 1, 'date': 1, 'stillageFileId': 1}"),
        @CompoundIndex(def = "{'clientId': 1, 'date': -1, 'stillageFileId': 1}"),

        // the same for userId
        @CompoundIndex(def = "{'userId': 1, 'date': 1, 'stillageFileId': 1}"),
        @CompoundIndex(def = "{'userId': 1, 'date': -1, 'stillageFileId': 1}")
})
public class File {

    @Id
    private String id;
    private String originalFileId;

    /**
     * Not null for files created from {@link ru.yandex.canvas.model.stock.StockFile} using
     * {@link ru.yandex.canvas.controllers.FilesController#uploadFile} method
     */
    private String stockFileId;

    /**
     * if file is loaded in context of idea
     * {@link IdeaDocument}
     */
    private String ideaId;

    private String name;

    private String stillageFileId;
    private StillageFileInfo stillageFileInfo;

    private String url;
    private String previewUrl;
    private String thumbnailUrl;
    private String largePreviewUrl;
    private String cropSourceUrl;
    private CropParameters cropParameters;
    private TurboParameters turboParameters;
    private Boolean isFromFeed;

    private Date date;

    // Either userId or clientId should be specified
    private Long clientId;
    private Long userId;

    @JsonIgnore
    private boolean archive;

    public String getId() {
        return id;
    }

    public void setId(String id) {
        this.id = id;
    }

    @Nullable
    public String getOriginalFileId() {
        return originalFileId;
    }

    public void setOriginalFileId(@Nullable String originalFileId) {
        this.originalFileId = originalFileId;
    }

    @Nullable
    public String getStockFileId() {
        return stockFileId;
    }

    public void setStockFileId(@Nullable String stockFileId) {
        this.stockFileId = stockFileId;
    }

    public String getUrl() {
        return url;
    }

    public void setUrl(String url) {
        this.url = url;
    }

    public String getPreviewUrl() {
        return previewUrl;
    }

    public void setPreviewUrl(String previewUrl) {
        this.previewUrl = previewUrl;
    }

    public String getThumbnailUrl() {
        return thumbnailUrl;
    }

    public void setThumbnailUrl(String thumbnailUrl) {
        this.thumbnailUrl = thumbnailUrl;
    }

    public String getLargePreviewUrl() {
        return largePreviewUrl;
    }

    public void setLargePreviewUrl(String largePreviewUrl) {
        this.largePreviewUrl = largePreviewUrl;
    }

    @Nullable
    public String getName() {
        return name;
    }

    public void setName(String name) {
        this.name = name;
    }

    public String getStillageFileId() {
        return stillageFileId;
    }

    public void setStillageFileId(String stillageFileId) {
        this.stillageFileId = stillageFileId;
    }

    public StillageFileInfo getStillageFileInfo() {
        return stillageFileInfo;
    }

    public void setStillageFileInfo(StillageFileInfo stillageFileInfo) {
        this.stillageFileInfo = stillageFileInfo;
    }

    public Optional<Long> getClientId() {
        return Optional.ofNullable(clientId);
    }

    public void setClientId(Long clientId) {
        this.clientId = clientId;
    }

    public Optional<Long> getUserId() {
        return Optional.ofNullable(userId);
    }

    public void setUserId(Long userId) {
        this.userId = userId;
    }

    public Date getDate() {
        return date;
    }

    public void setDate(Date date) {
        this.date = date;
    }

    public boolean isArchive() {
        return archive;
    }

    public void setArchive(boolean archive) {
        this.archive = archive;
    }

    public CropParameters getCropParameters() {
        return cropParameters;
    }

    public void setCropParameters(CropParameters cropParameters) {
        this.cropParameters = cropParameters;
    }

    public String getIdeaId() {
        return ideaId;
    }

    public void setIdeaId(String ideaId) {
        this.ideaId = ideaId;
    }

    public String getCropSourceUrl() {
        return cropSourceUrl == null ? url : cropSourceUrl;  // cropSourceUrl defaults to just url for older files
    }

    public void setCropSourceUrl(String cropSourceUrl) {
        this.cropSourceUrl = cropSourceUrl;
    }

    public TurboParameters getTurboParameters() {
        return turboParameters;
    }

    public void setTurboParameters(TurboParameters turboParameters) {
        this.turboParameters = turboParameters;
    }

    public Boolean getIsFromFeed() {
        return isFromFeed;
    }

    public void setIsFromFeed(Boolean isFromFeed) {
        this.isFromFeed = isFromFeed;
    }

    @Override
    public String toString() {
        return "File{" +
                "id='" + id + '\'' +
                ", originalFileId='" + originalFileId + '\'' +
                ", stockFileId='" + stockFileId + '\'' +
                ", name='" + name + '\'' +
                ", stillageFileId='" + stillageFileId + '\'' +
                ", stillageFileInfo=" + stillageFileInfo +
                ", url='" + url + '\'' +
                ", previewUrl='" + previewUrl + '\'' +
                ", thumbnailUrl='" + thumbnailUrl + '\'' +
                ", largePreviewUrl='" + largePreviewUrl + '\'' +
                ", cropSourceUrl='" + cropSourceUrl + '\'' +
                ", cropParameters=" + cropParameters +
                ", isFromFeed=" + isFromFeed +
                ", ideaId=" + ideaId +
                ", date=" + date +
                ", clientId=" + clientId +
                ", userId=" + userId +
                ", archive=" + archive +
                '}';
    }
}
