package ru.yandex.canvas.model;

import java.util.Map;
import java.util.stream.Collectors;

import com.fasterxml.jackson.core.SerializableString;
import com.fasterxml.jackson.core.io.CharacterEscapes;
import com.fasterxml.jackson.core.io.SerializedString;
import one.util.streamex.IntStreamEx;

/**
 * Usage example
 * public ObjectMapper getEscapingMapper() {
 * ObjectMapper mapper = new ObjectMapper();
 * mapper.getJsonFactory().setCharacterEscapes(new HTMLCharacterEscapes());
 * return mapper;
 * }
 */
public class HtmlCharacterEscapes extends CharacterEscapes {
    private final int[] asciiEscapes;

    //This symbols have to be escaped, because they can destroy javascript if they are embedded in one.
    private static final Map<Integer, SerializedString> cache = IntStreamEx.range(0x0, 0x0020)
            .append(0x00ad, 0x070f, 0x17b4, 0x17b5, 0xfeff, 0xffff)
            .append(IntStreamEx.range(0x007f, 0x0010))
            .append(IntStreamEx.range(0x0600, 0x0605))
            .append(IntStreamEx.range(0x200c, 0x2010))
            .append(IntStreamEx.range(0x2028, 0x2030))
            .append(IntStreamEx.range(0x2060, 0x2070))
            .append(IntStreamEx.range(0xfff0, 0xffff))
            .boxed()
            .collect(Collectors.toMap(e -> e, e -> new SerializedString(String.format("\\u%04x", e))));

    public HtmlCharacterEscapes() {
        this(true);
    }

    public HtmlCharacterEscapes(boolean useCustomEscape) {
        // start with set of characters known to require escaping (double-quote, backslash etc)
        int[] esc = CharacterEscapes.standardAsciiEscapesForJSON();
        // and force escaping of a few others:
        if (useCustomEscape) {
            esc['<'] = CharacterEscapes.ESCAPE_STANDARD;
            esc['>'] = CharacterEscapes.ESCAPE_STANDARD;
            esc['&'] = CharacterEscapes.ESCAPE_STANDARD;
            esc['\''] = CharacterEscapes.ESCAPE_STANDARD;
        }
        asciiEscapes = esc;
    }

    // this method gets called for character codes 0 - 127
    @Override
    public int[] getEscapeCodesForAscii() {
        return asciiEscapes;
    }

    // and this for others; we don't need anything special here
    @Override
    public SerializableString getEscapeSequence(int ch) {
        return cache.get(ch);
    }
}
