package ru.yandex.canvas.model.direct;

import java.io.IOException;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;
import java.util.stream.Stream;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonDeserializer;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import org.jetbrains.annotations.NotNull;

import ru.yandex.canvas.model.LowerCaseSerializer;

public class Privileges {

    private final Set<Permission> permissions = new HashSet<>();

    @JsonCreator
    public Privileges(@JsonProperty("permissions") Collection<Permission> permissions) {
        this.permissions.addAll(permissions);
    }

    public boolean checkPermissionOn(@NotNull Permission permission) {
        return permissions.contains(permission);
    }

    @SuppressWarnings("unused")
    public Set<Permission> getPermissions() {
        return permissions;
    }

    @JsonSerialize(using = LowerCaseSerializer.class)
    @JsonDeserialize(using = LowerCaseDeserializer.class)
    public enum Permission {
        /**
         * User permitted only to see creative preview
         */
        PREVIEW("Preview rights are required"),

        /**
         * User permitted to see creatives. Need for support crew
         */
        CREATIVE_GET("View creative rights are required"),

        /**
         * User permitted to create creatives
         */
        CREATIVE_CREATE("Write creative rights are required"),

        /**
         * User permitted to use idea facilities
         */
        IDEA("Idea rights are required"),

        /**
         * User permitted to use video addition facilities
         */
        VIDEO_ADDITION("Video addition rights are required"),

        /**
         * User permitted to use landings facilities
         */
        TURBO_LANDING("landing rights are required"),

        /**
         * User permitted to use desktop landing facilities
         */
        DESKTOP_LANDING("desktop landing rights are required"),

        /**
         * User permitted to use HTML5 facilities
         */
        HTML5_CREATIVES("HTML5 rights are required"),

        /**
         * Work with some user-files endpoint with userId instead of clientId is permitted
         */
        USER_FILES("User files rights are required"),

        /**
         * User permitted to use adaptive creatives https://st.yandex-team.ru/DIRECT-82119
         */
        ADAPTIVE_CREATIVES("adaptive creatives rights are required"),

        /**
         * forward compatibility https://st.yandex-team.ru/DIRECT-112804
         */
        UNKNOWN_PERMISSION("forward compatibility");

        private String restrictedMsg;

        Permission(String restrictedMsg) {
            this.restrictedMsg = restrictedMsg;
        }

        public String getRestrictedMsg() {
            return restrictedMsg;
        }
    }

    public static class LowerCaseDeserializer extends JsonDeserializer<Permission> {
        @Override
        public Permission deserialize(JsonParser p, DeserializationContext ctxt) throws IOException {
            String permissionAsAString = p.getValueAsString();
            return Stream.of(Privileges.Permission.values())
                    .filter(privilege -> privilege.name().toLowerCase().equals(permissionAsAString.toLowerCase()))
                    .findAny()
                    .orElse(Permission.UNKNOWN_PERMISSION);
        }
    }
}
