package ru.yandex.canvas.model.elements;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.validation.constraints.NotNull;

import com.fasterxml.jackson.annotation.JsonAnyGetter;
import com.fasterxml.jackson.annotation.JsonAnySetter;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonSubTypes;
import com.fasterxml.jackson.annotation.JsonSubTypes.Type;
import com.fasterxml.jackson.annotation.JsonTypeInfo;

import ru.yandex.adv.direct.media.CustomOption;
import ru.yandex.adv.direct.media.ElementOptions;
import ru.yandex.canvas.model.MediaSet;
import ru.yandex.canvas.model.validation.DifferentlyColoredOptions;
import ru.yandex.canvas.model.validation.presetbased.elements.ElementValidator;
import ru.yandex.canvas.service.rtbhost.ConstructorData.ConstructorDataElement;

import static ru.yandex.canvas.model.validation.DifferentlyColoredOptions.Metric.WGAG20;
import static ru.yandex.direct.utils.CommonUtils.nvl;

/**
 * @author skirsanov
 */
@JsonTypeInfo(use = JsonTypeInfo.Id.NAME, include = JsonTypeInfo.As.EXISTING_PROPERTY, property = "type", visible = true)
@JsonSubTypes({
        @Type(value = Button.class, name = ElementType.BUTTON),
        @Type(value = Description.class, name = ElementType.DESCRIPTION),
        @Type(value = Special.class, name = ElementType.SPECIAL),
        @Type(value = Disclaimer.class, name = ElementType.DISCLAIMER),
        @Type(value = AgeRestriction.class, name = ElementType.AGE_RESTRICTION),
        @Type(value = Headline.class, name = ElementType.HEADLINE),
        @Type(value = Domain.class, name = ElementType.DOMAIN),
        @Type(value = Image.class, name = ElementType.IMAGE),
        @Type(value = SecondImage.class, name = ElementType.SECOND_IMAGE),
        @Type(value = Logo.class, name = ElementType.LOGO),
        @Type(value = Legal.class, name = ElementType.LEGAL),
        @Type(value = Fade.class, name = ElementType.FADE),
        @Type(value = Video.class, name = ElementType.VIDEO),
        @Type(value = Phone.class, name = ElementType.PHONE),
        @Type(value = Subtitles.class, name = ElementType.SUBTITLES)
})
public abstract class Element {

    @NotNull
    private String type;

    private Boolean available;

    private String mediaSet;

    private List<ElementValidator> validators;

    public String getType() {
        return type;
    }

    public void setType(String type) {
        this.type = type;
    }

    public boolean getAvailable() {
        return available == null ? true : available;
    }

    public void setAvailable(Boolean available) {
        this.available = available;
    }

    public String getMediaSet() {
        return mediaSet;
    }

    public void setMediaSet(String mediaSet) {
        this.mediaSet = mediaSet;
    }

    public abstract Options getOptions();

    public void setValidators(List<ElementValidator> validators) {
        this.validators = validators;
    }

    /**
     * Validations to be used to validate element, if none are present in preset configuration
     */
    @JsonIgnore
    protected List<ElementValidator> getDefaultValidators() {
        return Collections.emptyList();
    }

    @JsonIgnore
    public List<ElementValidator> getEffectiveValidators() {
        return nvl(validators, getDefaultValidators());
    }

    /**
     * Maps current element to RTB-Host data.
     * <p>
     * Subclass should return Null if it does not want to contribute anything to RTB-Host export data.
     */
    @JsonIgnore
    public abstract ConstructorDataElement toRTBHostExportData(Map<String, MediaSet> mediaSets);

    public static abstract class Options {

        private Map<String, Object> additionalProperties = new HashMap<>();

        @JsonAnyGetter
        public Map<String, Object> getAdditionalProperties() {
            return this.additionalProperties;
        }

        @JsonAnySetter
        public void setAdditionalProperties(String name, Object value) {
            this.additionalProperties.put(name, value);
        }

        @JsonIgnore
        public ElementOptions toProto() {
            var builder = ElementOptions.newBuilder();
            fillProto(builder);
            additionalProperties.forEach((k, v) -> builder.addCustomOptions(
                    CustomOption.newBuilder().setName(k).setValue(v.toString()).build()));
            return builder.build();
        }

        protected abstract void fillProto(ElementOptions.Builder builder);
    }

    public static abstract class ColoredTextOptions extends Options implements OptionsWithPlaceholder, OptionsWithColor {

        private String placeholder;

        /**
         * All text element options have content as main text.
         */
        public abstract String getContent();

        public abstract void setContent(String content);

        public String getPlaceholder() {
            return placeholder;
        }

        public void setPlaceholder(@Nonnull String placeholder) {
            this.placeholder = placeholder;
        }

        @JsonIgnore
        public abstract Collection<String> getTexts();

        @Override
        protected void fillProto(ElementOptions.Builder builder) {
            var content = getContent();
            if (content != null) {
                builder.setContent(content);
            }

            if (placeholder != null) {
                builder.setPlaceholder(placeholder);
            }

            var color = getColor();
            if (color != null) {
                builder.setColor(color);
            }
        }
    }

    @DifferentlyColoredOptions(distance = 1.5, metric = WGAG20)
    public static abstract class ColoredTextOptionsWithBackground extends ColoredTextOptions {

        public abstract String getBackgroundColor();

        public abstract void setBackgroundColor(String backgroundColor);

        @Override
        protected void fillProto(ElementOptions.Builder builder) {
            super.fillProto(builder);
            var bkColor = getBackgroundColor();
            if (bkColor != null) {
                builder.setBackgroundColor(bkColor);
            }
        }
    }
}
