package ru.yandex.canvas.model.elements;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import javax.annotation.Nullable;
import javax.validation.Valid;
import javax.validation.constraints.NotNull;
import javax.validation.constraints.Positive;

import ru.yandex.canvas.model.MediaSet;
import ru.yandex.canvas.model.validation.presetbased.elements.ElementValidator;
import ru.yandex.canvas.model.validation.presetbased.elements.NotWhitespaceValidator;
import ru.yandex.canvas.model.validation.presetbased.elements.TextLengthValidator;
import ru.yandex.canvas.model.validation.presetbased.elements.ValidColorValidator;
import ru.yandex.canvas.model.validation.presetbased.elements.ValidUnicodeSymbolsValidator;
import ru.yandex.canvas.service.rtbhost.ConstructorData.ConstructorDataElement;

/**
 * @author skirsanov
 */
public class Headline extends Element implements ElementWithMaxLength {
    private static final int HEADLINE_MAX_LENGTH = 60;

    @Valid
    private Options options;

    @Nullable
    @Positive
    private Integer maxLength;

    public int getMaxLength() {
        return Optional.ofNullable(maxLength).orElse(HEADLINE_MAX_LENGTH);
    }

    public void setMaxLength(int maxLength) {
        this.maxLength = maxLength;
    }

    @Override
    public Options getOptions() {
        return options;
    }

    public void setOptions(Options options) {
        this.options = options;
    }

    @Override
    public List<ElementValidator> getDefaultValidators() {
        return List.of(
                new ValidUnicodeSymbolsValidator(),
                new ValidColorValidator(),
                new NotWhitespaceValidator(),
                new TextLengthValidator(getMaxLength())
        );
    }

    @Override
    public ConstructorDataElement toRTBHostExportData(Map<String, MediaSet> mediaSets) {
        return new ConstructorDataElement(getType(), options.getColor(), options.getContent());
    }

    @Override
    public String toString() {
        return "Headline{" +
                "options=" + options +
                '}';
    }

    public static class Options extends Element.ColoredTextOptions {
        private String content;
        private String color;

        public Options() {
        }

        public Options(@NotNull String content, String color) {
            this.content = content;
            this.color = color;
        }

        @Override
        public String getContent() {
            return content;
        }

        @Override
        public void setContent(String content) {
            this.content = content;
        }

        @Override
        public String getColor() {
            return color;
        }

        @Override
        public void setColor(String color) {
            this.color = color;
        }

        @Override
        public Collection<String> getTexts() {
            return Collections.singleton(content);
        }

        @Override
        public String toString() {
            return "Options{" +
                    "content='" + content + '\'' +
                    ", color='" + color + '\'' +
                    '}';
        }
    }
}
