package ru.yandex.canvas.model.html5;

import java.io.IOException;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.fasterxml.jackson.annotation.JsonProperty;
import org.springframework.data.annotation.Id;
import org.springframework.data.annotation.Transient;
import org.springframework.data.mongodb.core.mapping.Document;
import org.springframework.data.mongodb.core.mapping.Field;

import ru.yandex.canvas.model.CreativeWithScreenshot;
import ru.yandex.canvas.model.stillage.StillageFileInfo;
import ru.yandex.canvas.repository.mongo.MongoField;
import ru.yandex.canvas.service.html5.Html5Zip;

import static ru.yandex.canvas.repository.mongo.MongoField.ofFieldPath;

@Document(collection = "html5_sources")
public class Source implements CreativeWithScreenshot {

    public static final String ID_FIELD_NAME = "_id";
    public static final String CLIENT_ID_FIELD_NAME = "client_id";
    public static final String ARCHIVE_FIELD_NAME = "archive";
    public static final String SCREENSHOT_URL_FIELD_NAME = "screenshot_url";
    public static final String HTML_URL_FIELD_NAME = "html_url";
    public static final String HTML_REPLACEMENTS_NAME = "html_replacements";
    public static final String EXPANDED_HTML_URL_NAME = "expanded_html_url";
    public static final String SCREENSHOT_IS_DONE_FIELD_NAME = "screenshot_is_done";
    public static final String DATE_FIELD_NAME = "date";


    public static final MongoField ID = ofFieldPath(ID_FIELD_NAME);
    public static final MongoField CLIENT_ID = ofFieldPath(CLIENT_ID_FIELD_NAME);
    public static final MongoField ARCHIVE = ofFieldPath(ARCHIVE_FIELD_NAME);
    public static final MongoField HTML_URL = ofFieldPath(HTML_URL_FIELD_NAME);
    public static final MongoField SCREENSHOT_URL = ofFieldPath(SCREENSHOT_URL_FIELD_NAME);
    public static final MongoField SCREENSHOT_IS_DONE = ofFieldPath(SCREENSHOT_IS_DONE_FIELD_NAME);

    @Id
    protected String id;

    @Field("client_id")
    @JsonProperty("client_id")
    private Long clientId;

    private Date date;

    private String name;
    private Boolean archive;

    private Integer width;
    private Integer height;

    @Field("preset_id")
    private Integer presetId;

    @Field(SCREENSHOT_URL_FIELD_NAME)
    @JsonProperty(SCREENSHOT_URL_FIELD_NAME)
    private String screenshotUrl;

    @Field(SCREENSHOT_IS_DONE_FIELD_NAME)
    @JsonProperty(SCREENSHOT_IS_DONE_FIELD_NAME)
    private Boolean screenshotIsDone;

    @Field("preview_url")
    @JsonProperty("preview_url")
    private String previewUrl;

    private String url;

    @Field("stillage_info")
    @JsonProperty("stillage_info")
    private ZipStillageInfo stillageInfo;

    @Field("base_path")
    @JsonProperty("base_path")
    private String basePath;

    @Field("html_filename")
    @JsonProperty("html_filename")
    private String htmlFilename;

    @Field("html_url")
    @JsonProperty("html_url")
    private String htmlUrl;

    @Field(EXPANDED_HTML_URL_NAME)
    @JsonProperty(EXPANDED_HTML_URL_NAME)
    private String expandedHtmlUrl;

    @Field("screenshot_url_main_banner")
    @JsonProperty("screenshot_url_main_banner")
    private String screenshotUrlMainBanner;//основной баннер для расхлопа. Нужен отдельный скриншот

    @Field("bgrcolor")
    @JsonProperty("bgrcolor")
    private String bgrcolor;

    @Field("video_url")
    @JsonProperty("video_url")
    private String videoUrl;

    @Field("video_duration")
    @JsonProperty("video_duration")
    private Double videoDuration;

    @Field(HTML_REPLACEMENTS_NAME)
    @JsonProperty(HTML_REPLACEMENTS_NAME)
    private List<List<String>> htmlReplacements;

    @Field("source_image_info")
    @JsonProperty("source_image_info")
    private ImageStillageInfo sourceImageInfo;

    @Field("validation_status")
    @JsonProperty("validation_status")
    private CheckStatus validationStatus;

    @Field("invalid_path")
    @JsonProperty("invalid_path")
    private String invalidPath;

    public ImageStillageInfo getSourceImageInfo() {
        return sourceImageInfo;
    }

    public Source setSourceImageInfo(ImageStillageInfo sourceImageInfo) {
        this.sourceImageInfo = sourceImageInfo;
        return this;
    }

    @Transient
    private byte[] archiveContent;

    public byte[] getArchiveContent() {
        return archiveContent;
    }

    public Html5Zip unzipArchiveContent() throws IOException {
        if (getArchiveContent() != null) {
            return new Html5Zip(getArchiveContent());
        } else {
            throw new IOException("Archive should be downloaded first!");
        }
    }

    public void setArchiveContent(byte[] archiveContent) {
        this.archiveContent = new byte[archiveContent.length];
        System.arraycopy(archiveContent, 0, this.archiveContent, 0, archiveContent.length);
    }

    public String getId() {
        return id;
    }

    public Source setId(String id) {
        this.id = id;
        return this;
    }

    public Long getClientId() {
        return clientId;
    }

    public Source setClientId(Long clientId) {
        this.clientId = clientId;
        return this;
    }

    public Date getDate() {
        return date;
    }

    public Source setDate(Date date) {
        this.date = Date.from(date.toInstant());
        return this;
    }

    public String getName() {
        return name;
    }

    public Source setName(String name) {
        this.name = name;
        return this;
    }

    public Boolean getArchive() {
        return archive;
    }

    public Source setArchive(Boolean archive) {
        this.archive = archive;
        return this;
    }

    public Integer getWidth() {
        return width;
    }

    public Source setWidth(Integer width) {
        this.width = width;
        return this;
    }

    public Integer getHeight() {
        return height;
    }

    public Source setHeight(Integer height) {
        this.height = height;
        return this;
    }

    public String getScreenshotUrl() {
        return screenshotUrl;
    }

    public Source setScreenshotUrl(String screenshotUrl) {
        this.screenshotUrl = screenshotUrl;
        return this;
    }

    public Boolean getScreenshotIsDone() {
        return screenshotIsDone;
    }

    public Source setScreenshotIsDone(Boolean screenshotIsDone) {
        this.screenshotIsDone = screenshotIsDone;
        return this;
    }

    public String getPreviewUrl() {
        return previewUrl;
    }

    public Source setPreviewUrl(String previewUrl) {
        this.previewUrl = previewUrl;
        return this;
    }

    public String getUrl() {
        return url;
    }

    public Source setUrl(String url) {
        this.url = url;
        return this;
    }

    public ZipStillageInfo getStillageInfo() {
        return stillageInfo;
    }

    public Source setStillageInfo(ZipStillageInfo stillageInfo) {
        this.stillageInfo = stillageInfo;
        return this;
    }

    public String getBasePath() {
        return basePath;
    }

    public Source setBasePath(String basePath) {
        this.basePath = basePath;
        return this;
    }

    public String getHtmlFilename() {
        return htmlFilename;
    }

    public Source setHtmlFilename(String htmlFilename) {
        this.htmlFilename = htmlFilename;
        return this;
    }

    public String getHtmlUrl() {
        return htmlUrl;
    }

    public Source setHtmlUrl(String htmlUrl) {
        this.htmlUrl = htmlUrl;
        return this;
    }

    public String getExpandedHtmlUrl() {
        return expandedHtmlUrl;
    }

    public Source setExpandedHtmlUrl(String expandedHtmlUrl) {
        this.expandedHtmlUrl = expandedHtmlUrl;
        return this;
    }

    public String getScreenshotUrlMainBanner() {
        return screenshotUrlMainBanner;
    }

    public Source setScreenshotUrlMainBanner(String screenshotUrlMainBanner) {
        this.screenshotUrlMainBanner = screenshotUrlMainBanner;
        return this;
    }

    public String getBgrcolor() {
        return bgrcolor;
    }

    public Source setBgrcolor(String bgrcolor) {
        this.bgrcolor = bgrcolor;
        return this;
    }

    public String getVideoUrl() {
        return videoUrl;
    }

    public Source setVideoUrl(String videoUrl) {
        this.videoUrl = videoUrl;
        return this;
    }

    public Double getVideoDuration() {
        return videoDuration;
    }

    public Source setVideoDuration(Double videoDuration) {
        this.videoDuration = videoDuration;
        return this;
    }

    public List<List<String>> getHtmlReplacements() {
        return htmlReplacements;
    }

    public Source setHtmlReplacements(List<List<String>> htmlReplacements) {
        this.htmlReplacements = htmlReplacements;
        return this;
    }

    @Override
    public String fetchScreenshotUrl() {
        return getScreenshotUrl();
    }

    @Override
    public Boolean fetchScreenshotIsDone() {
        return getScreenshotIsDone();
    }

    public Integer getPresetId() {
        return presetId;
    }

    public Source setPresetId(Integer presetId) {
        this.presetId = presetId;
        return this;
    }

    public static class ImageStillageInfo extends ZipStillageInfo {
        private Map<String, Object> metadataInfo = new HashMap<>();

        public Map<String, Object> getMetadataInfo() {
            return metadataInfo;
        }

        public ImageStillageInfo setMetadataInfo(Map<String, Object> metadataInfo) {
            this.metadataInfo = metadataInfo;
            return this;
        }

        public ImageStillageInfo() {
            super();
        }

        public ImageStillageInfo(StillageFileInfo stillageFileInfo) {
            super(stillageFileInfo);
            this.setMetadataInfo(stillageFileInfo.getMetadataInfo());
        }
    }


    public static class ZipStillageInfo {
        @Field("id")
        private String id;

        private String url;

        private int fileSize;

        private String md5Hash;

        private String mimeType;

        private String contentGroup;

        public String getId() {
            return id;
        }

        public ZipStillageInfo setId(String id) {
            this.id = id;
            return this;
        }

        public String getUrl() {
            return url;
        }

        public ZipStillageInfo setUrl(String url) {
            this.url = url;
            return this;
        }

        public int getFileSize() {
            return fileSize;
        }

        public ZipStillageInfo setFileSize(int fileSize) {
            this.fileSize = fileSize;
            return this;
        }

        public String getMd5Hash() {
            return md5Hash;
        }

        public ZipStillageInfo setMd5Hash(String md5Hash) {
            this.md5Hash = md5Hash;
            return this;
        }

        public String getMimeType() {
            return mimeType;
        }

        public ZipStillageInfo setMimeType(String mimeType) {
            this.mimeType = mimeType;
            return this;
        }

        public String getContentGroup() {
            return contentGroup;
        }

        public ZipStillageInfo setContentGroup(String contentGroup) {
            this.contentGroup = contentGroup;
            return this;
        }

        public ZipStillageInfo() {

        }

        public ZipStillageInfo(StillageFileInfo stillageFileInfo) {
            this.setId(stillageFileInfo.getId())
                    .setContentGroup(stillageFileInfo.getContentGroup())
                    .setFileSize(stillageFileInfo.getFileSize())
                    .setMd5Hash(stillageFileInfo.getMd5Hash())
                    .setUrl(stillageFileInfo.getUrl())
                    .setMimeType(stillageFileInfo.getMimeType());
        }

    }

    public CheckStatus getValidationStatus() {
        return validationStatus;
    }

    public Source setValidationStatus(CheckStatus validationStatus) {
        this.validationStatus = validationStatus;
        return this;
    }

    public String getInvalidPath() {
        return invalidPath;
    }

    public Source setInvalidPath(String invalidPath) {
        this.invalidPath = invalidPath;
        return this;
    }
}
