package ru.yandex.canvas.model.html_builder;

import java.io.StringWriter;

import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.github.mustachejava.Mustache;
import com.google.common.collect.ImmutableMap;

import ru.yandex.canvas.model.CreativeData;
import ru.yandex.canvas.model.CreativeDocument;
import ru.yandex.canvas.model.HtmlCharacterEscapes;

public abstract class AbstractHtmlCreativeWriter implements HtmlFromCreativeWriter {

    private final ObjectMapper asciiEscapedMapper = new ObjectMapper();

    // true - html строится для rtb host'а; false - html строится для превью
    private final boolean isHtmlForRtbHost;

    private final Mustache template;

    public AbstractHtmlCreativeWriter(boolean isHtmlForRtbHost, Mustache template) {
        this.isHtmlForRtbHost = isHtmlForRtbHost;
        this.template = template;
        asciiEscapedMapper.getFactory()
                .configure(JsonGenerator.Feature.ESCAPE_NON_ASCII, true)
                .setCharacterEscapes(new HtmlCharacterEscapes());
    }

    protected String getAnyField(CreativeDocument creativeDocument) {
        try {
            return asciiEscapedMapper.writeValueAsString(getVisibleCreativeData(creativeDocument.getData()));
        } catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }
    }

    protected String getStyleField(CreativeDocument creativeDocument) {
        return getBodyStyleForCreative(getVisibleCreativeData(creativeDocument.getData()));
    }

    protected String getHtmlStyleAttr(CreativeDocument creativeDocument) {
        return getHtmlStyleAttrForCreative(getVisibleCreativeData(creativeDocument.getData()));
    }

    /**
     * Возвращает css-стиль (высота и ширина) для креатива.
     *
     * @param creativeData данные креатива
     * @return css-стиль
     */
    private String getBodyStyleForCreative(CreativeData creativeData) {
        if (!creativeData.getOptions().getIsAdaptive()) {
            return "";
        }

        if (!isHtmlForRtbHost) {
            // Для превью адаптивных креативов подставляем размеры в стили body, чтобы креатив отобразился с размерами,
            // отличными от 0x0.
            return String.format("width:%dpx;height:%dpx", creativeData.getWidth(), creativeData.getHeight());
        } else {
            return "width:100%;height:100%;";
        }
    }

    /**
     * Возвращает атрибут с css-стилем для креатива.
     *
     * @param creativeData данные креатива
     * @return css-стиль
     */
    private String getHtmlStyleAttrForCreative(CreativeData creativeData) {
        //Для RTB адаптивных креативов в html теге нужно прописать стили
        if (creativeData.getOptions().getIsAdaptive() && isHtmlForRtbHost) {
            return " style=\"width:100%;height:100%;\"";
        }

        return "";
    }

    /**
     * Возвращает видимые данные креатива.
     *
     * @param creativeData все данные креатива
     * @return видимые данные креатива
     */
    private CreativeData getVisibleCreativeData(CreativeData creativeData) {
        creativeData = creativeData.onlyVisibleData();

        // Для адаптивных креативов не нужно проставлять width и height, т.к. размеры для таких креативов определяются
        // на стороне БК.
        if (creativeData.getOptions().getIsAdaptive() && isHtmlForRtbHost) {
            creativeData
                    .withWidth(null)
                    .withHeight(null);
        }

        return creativeData;
    }

    @Override
    public String composeCreativeHTML(CreativeDocument creativeDocument) {

        StringWriter writer = new StringWriter();
        ObjectMapper asciiEscapedMapper = new ObjectMapper();
        asciiEscapedMapper.getFactory().configure(JsonGenerator.Feature.ESCAPE_NON_ASCII, true)
                .setCharacterEscapes(new HtmlCharacterEscapes());

        template.execute(writer,
                ImmutableMap.of(
                        "any", getAnyField(creativeDocument),
                        "style", getStyleField(creativeDocument),
                        "html_style_attr", getHtmlStyleAttr(creativeDocument)
                ));

        return writer.toString();
    }
}
